import arcpy
from arcpy.sa import *
import numpy as np
import numpy.ma as ma
from scipy.stats.mstats import mquantiles
import pandas as pd
import shelve
import os
import sys
import re
from math import isclose
import fnmatch
import glob
from datetime import datetime, timedelta
from dateutil.relativedelta import relativedelta
import xml.etree.ElementTree as ET
import calendar
import traceback
import warnings
import codecs
from osgeo import gdal
# from guppy import hpy

'''
Description: NOAA/NOS/NCCOS/CCMA/COAST Remote Sensing tools

Author: Andrew Meredith

Date: 03-15-2015

Change log:
v0.8.10 am
 - Raster statistics by polygon: removed x,y values from csv header as we're not including geospatial information in the output file
v0.8.20 am  03/25/2015
 - Fixed bug in Time Series Composite if input rasters small (only single block then no mosaicking required).
v0.9.0 am  03/27/2015
- Time series compositing
    - added percentile statistic options
    - changed median to use new (50-) percentile functionality as the median routine i was using does not work with ArcGIS 10.2.0
v0.9.1 am  03/30/2015
- PixelExtractByPoint
    - fixed bug in extraction of feature layer filename
v0.9.2 am  03/30/2015
- PixelExtractByPolygon
    - fixed bug if no field attribute selected
v0.9.3 am  03/30/2015
- set arcpy.env.overwriteOutput = True
- RasterStatsByPolygon
    - changed filename included in CSV output file to input filename
    - handled comma in pt field value
- PixelExtractByPolygon
    - set snapRaster to force polygon cell alignment to input raster
- MaskNoData
    - supported writing rasters to GDB
- Added some input validation within ToolValidator classes
v0.9.4 am  03/30/2015
    - fixed bug in identifying arcgis version for "median" command switching
v0.9.5 am 04/15/2015
    - Changed to get all files using da.Walk not da.Listraster as problem if raster also listed in TOC
v0.9.6 am  04/15/2015
    - Pixel extract by points
        - date matchup only uses date portion of the filename and date field since ArcGIS shapefiles only support date not date/time
        - supported "allowed time difference" of 0
v0.9.7 am  04/16/2015
    - Time series composite
        - manage memory usage based on number of files we're compositing
v0.9.8 am  04/22/2015
    - Pixel extract by points
        - added optional matchup parameter "Field data value attribute"
        - name of attribute is stored in CSV header
        - date matchup uses datetime if units = 'hour(s)'
    - Time series composite
        - changed handling of output dtype
        - changed naming of statistic for Q-PERCENTILE to QPERCENTILE as GDB's don't like '-' in filename
v0.9.9 am  04/23/2015
    - Changed from specifying individual ignore data values to specifying a min/max valid data range
        - updated Time series composite & Mask no data values
        - added support in Pixel extract by points
    - Time series composite
        - resolved rename error b/c deleting of existing file with delete_management failed
v0.9.10 am  04/23/2015
    - Time series composite and Mask No data
        - adds color table from input raster to output raster if data type is "integer"
        - changed handling of output dtype (again)
v1.0.0 am   05/01/2015
    - Final CA workshop version
        - debugging off
v1.1.0 am
    - PE extract and Raster statistic tools
        - "date" field now represents the mid point of start/end date for L4 type input files
    - Time Series composite
        - fixed handling of varying 'required' / 'optional' input fields based on "composite type"
        - added "Do not span months" option
        - added "copy landmask" option
        - modified output (L4) filename format
            - <prefix>_yyyy.mmdd_mmdd.L4.<area>.<prod>.<statistic>[_<suffix>].tif
        - if input files are L4 type files, the selection of files for a composite period is now based on the mid point of start/end date of each L4 file (previously used the start date)
    - "nodata value" validation
    - PE extract by point & polygon
        - "FID" & "Shape"  excluded from field selection list
        - added column to CSV with the unscaled product value
    - Raster statistics extract by  polygon
        - added column(s) to CSV with the unscaled product value
        - added support to dynamically mask valid pixels based on user defined min/max range
v1.1.1 am
    - Disabled my debug stuff (i.e. import guppy)
v1.1.2 am
    - fixed bug if min_valid_val = 0
v1.1.3 am
    - PE extract by point
        - fixed bug with multi-band input
        - added band_num column
v1.1.4 am
    - PE extract by point
        - identified ArcGIS memory leak with ExtractMultiValuesToPoints that had potential to cause invalid output to csv
            - developed custom point extract routine
        - if do_matchup, pre-filtered file list based on dates of points in feature class
v1.1.5 am 7/7/2015
    - PE extract by point
        - spatial reference of the X,Y values written to the CSV file set to lon/lat (WGS84)
v1.1.7 am 7/7/2015
    - Time Series composite
        - Fixed bug in Copy Landmask and forced to 8bit unsigned tif
v2.0.0 am 8/21/2015
    - Raster statistics by  polygon
        - restricted zone data field types to valid types
        - improved exception handler
        - fixed bug in handling of nodata
v2.0.1 am 8/24/2015
    - Raster statistics by  polygon
        - fixed bug in handling of nodata
v2.0.2 am 9/1/2015
    - Pixel extract by polygon
        - fixed "bug" when reading data from linux management drive
v2.0.3 am 9/8/2015
    - fixed error in parse_fn when dealing with 32-bit tifs
    - Raster statistics by polygon
        - fixed "bug" when reading data from linux management drive
    - Distributed CA version
        - debugging off
v2.1.0 am 9/24/2015
    - defined some default arcpy.env settings
    - Pixel extract by point
        - Added ability to extract pixels within XxX window centered on point features
v2.1.1      am 10/6/2015
    - Pixel extract by point
        - Trapped missing "Optional" required parameters in ToolValidator
        - Added "OID" field type to Point attribute and Field data attribute filter list
        - Trapped NULL date error in point FC and produced warning
    - Time Series composite
        - Fixed bug if both Create Count and Copy Landmask options selected
    - Pixel extract by polygon
        - fixed bug if no prod_name identified in filename
    - All tools
        - Added File selection method option to support alternate method (i.e. filelist) to specify input files for processing
v2.1.1b am 1/14/2016
    - Time Series composite
        - Added option to combined files across years
v2.1.2
    - Raster statistics by polygon
        - Fixed error if "median" statistic selected by forcing the pixel type of the masked raster to match the input raster
v2.1.3b

 Added GenerateAnomaly tool.
v2.1.3 am 1/14/2016
    - Time Series composite
        - Added option to combined files across years
v2.1.4 am 4/6/2016
    - Pixel extract by polygon
        - Fixed bug to handle feature class names beginning with number
    - Changes to support update in SAPS filenaming
    - Updates to support specification of alternate filename parsing definition (alpha)
        - includes new tool "Define filename parsing format"
 Fixed bug FC names start with nummber; updates SAPS filename changes; updates alternate FN parsing definition.
    - New tools "Generate anomaly rasters" (alpha)
 Documentation updates.

 v2.1.5 am 6/22/2016
    - Added ApplyCorrection Tool
    - Fixed bug in RasterStatsByPolygon if no min/max filtering 
 v2.1.7 am 7/21/2016
    - Time series composite
        - Fixed bug that showed up in ArcGIS 10.4
        - Fixed bug in handling of composite_type == 'all'
 v2.1.7.01 am 9/2/2016
    - Fixed bug using scratchWorkspace in create_masks
 v2.1.8 am 9/2/2016
    - Additional error trapping
 v2.1.9 am 2/15/2017
    - fixed bug if input rasters cover multiple regions
    - added support for NASA L3 mapped CyAN filenames
 v2.1.11 am 8/7/2017
    - Pixel Extract - trapped bad geometery error
    - Implemented support for unicode
    - NASA L3 OLCI daily CyAN filename
    - Raster Statistics - Fixed issue with arc temporary ("t_*") files not being deleted
 v2.2.0 am 8/14/2017
    - Pixel Extract by point 
        - summarize pixels in 3x3 or 5x5 window
        - select multiple attributes for output from point file
        - if matchup option, added time delta field in CSV
        - 3x3 or 5x5 window include pixel as columns instead of rows 
        - band number filtering for multiband geotiffs
    - Pixel Extract by polygon
        - select multiple attributes for output from point file
 v2.2.1 am 8/30/2017
    - Change to support modification in NASA OLCI filenames
 v2.2.2 am 
    - Raster statistics - Handle invalid characters (e.g. '-') in the input raster filenames
    - Pixel Extract by polygon - fixed bug in writing attribute values to csv
    - Included 'prod_ver' column in output csv files if input filenames identified as SAPS type product names
    - Time series composite - performance improvements when input rasters located on network
 v2.2.3 am 11/2/2018
    - Pixel Extract by point - extract point selection off by half width of cell
    - Pixel Extract by polygon - error when selecting more than one attribute from feature class
 v2.3.0 am 1/11/2019
    - 'unscaled' column support for SAPS v2 CI products
        - in Raster statistics, Pixel Extract by point, Pixel Extract by polygon
        - excluded column if no 'unscaling' support       
    - CreateTimeSeries - Added tool command string to 'lineage' metadata in each output *.tif.xml 
 v2.3.1 am 5/21/2019
    - Added SwapColormap tool to replace colormap in rasters
 v2.3.2 am 5/21/2019
    - Raster Statistics - Added options to filter by matchup date in polygon shapefile
        - caveats 
            o only supporting same day matchups
            o zone field should return a unique polygon
            o does not honor the selection of features in "zone data" feature class
 v2.3.4 am 7/17/2020 
    - Raster Statistics - Zone Data feature class to layer to fix SelectLayerByAttribute_management (if running from script)
    - PixelExtractByPoint option to reformat output from multi-band raster so each value from each band is added to column "bnum_#" instead of the default which creates a new row for band. (requires pandas therefore ArcGIS v10.4+)
    - Raster Statistics - tweak to improve processing efficicency
    - Pixel Extract by point - quote string fields to avoid potential problems with commas
 v3.0.0 am 7/21/2020
    - Migrated to ArcGIS Pro
 v3.0.1 am 9/23/2020 
    - PixelExtractByPoint handled error of "NULL" date in point FC on matchup set
 V3.0.2 am 
    - MaskNoData (errors only in ArcGIS Pro version)
        o fixed bug in writing to wrong output location
        o fixed bug with scratchWorkspace 
        o OSError with arcpy.Describe('%s\\Band_1' % raster_fn) 
    - GenerateAnomaly - creates simple difference
 V3.0.3 am 10/6/2021
    - Raster Statistics - Fixed bug introduced in v2.3.4 that dropped the zone field column in output CSV
    - PixelExtractByPoint - Added additional pixel window extract sizes ('7x7', '9x9', '11x11', '13x13')
 V3.0.4 am 12/9/2021
    - Raster Statistics - If raster has no valid data pixels within zones, a row is added with results set to 0. Previously, only a warning message was displayed and no row added. 
 V3.1.0 am 06/10/2022
    - Fix for AttributeError: 'GPEnvironment' object has no attribute 'cell_size' introduced with ArcGIS Pro v2.9
    - Support for change to NOAA/NCCOS OLCI filenaming convention
    - Compositing NOAA/NCCOS products adds product version to composited filenames
    - Propogate NOAA/NCCOS SAPS metadata to composited products (if output workspace is FileSystem not GDB)
 V3.1.1 am 6/6/2023
    - Time Series Composite - added support for creating 32-bit float composite products when using NCCOS "real" input products
 V3.1.2 am 6/27/2023
    - Time Series Composite - fix for testing raster cellsize match that failed with geographic projection
 V3.1.3 am 7/21/2023
    - Extract tools - No unscaling of NCCOS "real" products
    - Extract tools - Added NCCOS v3 products to can_unscale
 V3.1.4 am 8/25/2023
    - Raster Statistics - Added "SUM_area_sqkm" attribute to output CSV if the input rasters are in a projected coordinate system and the selected "Statistic" is either "SUM" or "All"
    
'''

class InvalidBandNum(Exception):
    def __init__(self, value):
        self.value = value
        self.band_num, self.band_count = value.split(',')
    def __str__(self):
        return repr(self.value)

class missingRequiredParameter(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

class invalidParameter(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)
        
class filenameFormatError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

class overwriteError(Exception):
    pass

class LicenseError(Exception):
    pass

class RsTools(object):
    # Base class
    def __init__(self):
        
        arcpy.AddMessage('RS_tools version = %s\n' % self.getversion())        
        
        config_dir = os.path.join(os.path.dirname(__file__), 'config')
        if not os.path.exists(config_dir):
            os.mkdir(config_dir)

        config_fn = os.path.join(config_dir, 'rs_tools.db')
        self.config_dir = config_dir
        self.config_fn = config_fn
        
        #default SAPS L3 filename definition used to parse filename into components
        #   'delim' component delimiter (required)
        #   component definition is a list of 0, 1, or 3 items:
        #       part_desc = [] - component will be assigned None (only valid for area_name and prod_name)
        #       part_desc[0] = index within the list of components split on 'delim'
        #       part_desc[1] = starting index position within the string returned by part_desc[0]. 
        #                      if no value then no subsetting. (optional)
        #       part_desc[2] = ending index position within the string returned by part_desc[0]. 
        #                      if no value then no subsetting. (optional)
        self.rstools_default_fn_parser_def = {'name': 'RSTools_default',
                                       'delim': '.',
                                       'year': [1, 0, 4], 
                                       'mm': [2, 0, 2],
                                       'dd': [2, 2, 4],
                                       'hhmm': [3, 0, 4],
                                       'area_name': [5],
                                       'prod_name': [7],
                                       'ver_info': [6],
                                      }
        self.fn_parser_def = self.load_fn_parser_def()
        
        #list of product names that support unscaling pixel values
        #   - "_ta" is a hack to get around past poor choice for 
        #       filenaming that includes projection identifier suffix ("_ta" )

        self.known_prods = ['ci', 'cicyano', 'cinoncyano', 'ci-cicyano', 'ci_ta', 'cicyano_ta', 'cinoncyano_ta']
        
    def setup(self, tool_name):
        self.in_ws = None
        self.in_filter = None
        self.out_ws = None
        self.out_ws_type = None
        self.recursive = False
        self.lic_checked_out = {}
        self.orig_overwriteoutput = arcpy.env.overwriteOutput
        self.prod_name = None
        #used to replace '.' for filenames created in GDB
        self.ws_delimiter = '__'

        self.log_fn = "%s\\RStools_%s.log" % (arcpy.env.scratchFolder, tool_name)
        ##To enable debugging, set to "True"
        # self.debug = True
        self.debug = False
        if self.debug:
            self.log_file = codecs.open(self.log_fn, "w", encoding="utf-8")
            arcpy.AddMessage('Debugging on...debugging log will be created in %s.' % self.log_fn)
        else:
            self.log_file = None

        #Set environments
        arcpy.env.overwriteOutput = True
        arcpy.env.XYResolution = "0.00001 Meters"
        arcpy.env.rasterStatistics = 'NONE'
        arcpy.env.pyramid = 'NONE'
        arcpy.env.buildStatsAndRATForTempRaster = False

    def getversion(self):
        ''' Extract and return RS Tools version from script filename.'''
        scriptname = os.path.splitext(os.path.basename(__file__))[0]
        version = scriptname.split('_')[2] if scriptname.split('_')[2] else 'unknown'
        return version
        
    def get_arc_lic(self, extension_name):
        try:
            self.lic_checked_out[extension_name] = False
            if arcpy.CheckExtension(extension_name) == "Available":
                #arcpy.AddMessage("Getting the %s license" % extension_name)
                arcpy.CheckOutExtension(extension_name)
                self.lic_checked_out[extension_name] = True
            else:
                # raise a custom exception
                raise LicenseError

        except LicenseError:
            arcpy.AddError("%s license is unavailable" % extension_name)

    def return_arc_lic(self, extension_name):
        # arcpy.AddMessage("Returning %s license" % extension_name)
        arcpy.CheckInExtension(extension_name)

    def get_ws_type(self, ws):
        return arcpy.Describe(ws).workspaceType

    def get_out_ext(self, ws_type):
        '''
        Return extension name based on workspace type.
        '''
        #to folder or gdb
        if ws_type == 'FileSystem':
            return 'tif'
        else:
            return ''
            
    def load_fn_parser_def(self):
        '''
        Load a dictionary with the filename parser definition.
        
        Use the custom filename definition if defined otherwise the
        default RSTools definition is used.
        
        Return filename parser definiton dictionary
        '''
        config_fn = self.config_fn
        fn_parser_def = None
        
        if os.path.exists(config_fn):
            #load defaults from existing definition
            try:
                arcpy.AddMessage('Loading filename parser definition from %s' % config_fn)
                s = shelve.open(config_fn)
                if 'fn_parser_def' in s: 
                    fn_parser_def = s['fn_parser_def']
                    arcpy.AddMessage('fn_parser_def=%s' % fn_parser_def)
            finally:
                s.close()
                del s
                
        if fn_parser_def is None:
            arcpy.AddMessage('Loading RS_Tools default filename parser definition')
            fn_parser_def = self.rstools_default_fn_parser_def
            
        return fn_parser_def

    def validate_fn_prefix(self, prefix):
        '''
        Return prefix with '_' appended if not already included.
        '''
        if not prefix:
            return ''

        val_prefix = prefix if prefix[-1] == '_' else prefix + '_'
        return val_prefix

    def validate_fn_suffix(self, suffix):
        '''
        Return suffix with '_' prepended if not already included.
        '''
        if not suffix:
            return ''

        val_suffix = suffix if suffix[0] == '_' else '_' + suffix
        return val_suffix

    def validate_csv_name(self, fc_name):
        '''
        Return filename with .csv extension.
        '''
        fc_noext, fc_ext = os.path.splitext(fc_name)
        if fc_ext.lower() != '.csv':
            return '%s.csv' % fc_name
        else:
            return fc_name

    def get_files(self, ws=None, filter=None, recursive=None):
        '''
        Return list of rasters form workspace or list of files contained in a text file.
        '''

        files = []

        if self.input_method == 'Workspace':
            if not ws: ws = self.in_ws
            if not filter: filter = self.in_filter
            if not recursive: recursive = self.recursive

            ws_type = self.get_ws_type(ws)
            if self.debug: self.log_file.write ('in ws type=%s\n' % ws_type)

            if ws_type == 'FileSystem':

                walk = arcpy.da.Walk(ws, datatype="RasterDataset", type="All")

                for dirpath, dirnames, fns in walk:
                    for fn in fns:
                        if fnmatch.fnmatch(fn, filter):
                            #convert backslash to forwardslash in path to avoid issues in arcpy
                            files.append(os.path.join(dirpath, fn).replace('\\', '/'))
                    #only getting files from first level if not recursive
                    if not recursive: break
            else:
                arcpy.env.workspace = ws
                files = [os.path.join(ws, f) for f in arcpy.ListRasters(filter)]

        else:
            with open(self.in_filelist_fn, "r") as f:
                for line in f:
                    fn = line.strip()
                    if not fn or fn[0] == '#':
                        continue
                    if os.path.exists(fn):
                        files.append(fn)
                    else:
                        arcpy.AddWarning('File "%s" from filelist not found on disk...skipped.' % fn)

        if self.debug: self.log_file.write ('%s files found.\n' % len(files))
        if self.debug: self.log_file.write ('get_files: files = %s\n' % (files))

        return files

    def fn_info_by_date(self, files):
        '''
        Return dictionary with filenames keyed by datetime (object) extracted from filename.

        Filenames in non-standard format are skipped and excluded.
        '''
        fn_dict = {}
        for fn in files:
            (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(fn)
            if not year:
                arcpy.AddWarning('File skipped...Unable to parse filename "%s". Not in a standard SAPS L3/L4 format.' % fn)
                continue

            try:
                fn_dt = datetime.strptime( '%s%s%s %s' % (year, mm, dd, hhmm), '%Y%m%d %H%M')
            except:
                arcpy.AddError('Unable to parse date/time value from filename %s (y=%s m=%s d=%s hm=%s)' %  (fn, year, mm, dd, hhmm))
                raise
            if not fn_dt:
                arcpy.AddError('Unable to parse date/time value from filename %s (y=%s m=%s d=%s hm=%s)' %  (fn, year, mm, dd, hhmm))
                return None

            # if not fn_dt:
                # arcpy.AddWarning('File skipped...Invalid datetime values extracted from filename "%s". Not in a standard SAPS L3/L4 format.' % fn)
                # continue

            #allow for potential duplicate files for the same datetime
            while True:
                if fn_dt in fn_dict:
                    fn_dt += timedelta(microseconds=1)
                    arcpy.AddWarning('Datetime of %s matches other rasters. (date/time=%s). Check you are not compositing multiple product types into the single composite.' % (fn, fn_dt.strftime('%Y%m%d %H%M')))
                    if self.debug: self.log_file.write ('Warning: Datetime of %s matches other rasters. date/time=%s\n' % (fn, fn_dt.strftime('%Y%m%d %H%M')))
                else:
                    fn_dict[fn_dt] = fn
                    break

        return fn_dict

    def isNASA_fn(self, fn):
        '''
        Identify if NASA CyAN filename.
            - 'M' meris; 'L' OLCI; 'O' previous code for OLCI
        '''
        fn_path, fn_base = os.path.split(fn)
        return ((fn_base.startswith('M') or fn_base.startswith('O') or fn_base.startswith('L')) and '.L3m_' in fn_base)

    def isS2_MSI_fn(self, fn):
        '''
        Identify Sentinel-2 MSI L1C or L2A filename.
        '''
        fn_path, fn_base = os.path.split(fn)
        return (fn_base.startswith('S2A_MSI') or fn_base.startswith('S2B_MSI'))
        
    def parse_fn(self, fn):
        '''
        Parse a SAPS standard L3 filename into components.
            Examples: aqua.2014018.0118.1830S.L3.EF3.v670.chloc2.tif
                      aqua.2014018.0118.1830C.L3.EF3.v670.tif
                      aqua.2014046.0215.1851_2030C.L3.GM1.v670.tif
        or L4 filename into components
            Example: <prefix>_2009.0101_0110.L4.<product>.MAXIMUM[_<suffix>][.tif]
        **
        NASA L3 mapped:
            Examples: 
                daily       - M2011184.L3m_DAY_CYANF_CI_cyano_CYAN_CONUS_300m_7_5.tif
                7day bin    - M20071832007189.L3m_7D_CYAN_CI_cyano_CYAN_CONUS_300m_1_4.tif
        **            
        Supports both SAPS standard filenames (as above) and modified SAPS filenames meeting
           requirements for ArcGIS GDB (i.e. '.' delimiter replaced by self.ws_delimiter)
           NOTE: potential issues identify product name for GDB compliant filenames.
            L3
            envisat.2011003.0103.1744C.L3.CAC.v670_1.tif
            envisat.2011003.0103.1744C.L3.CAC.v670.CI.tif
            envisat__2011003__0103__1744C__L3__CAC__v670__CIcyano
            envisat__2011003__0103__1744C__L3__CAC__v670__CI
            L4
            ts_2011.0101_0110.L4.CAC.CI.MAXIMUM_10day_new.tif
            ts_2011-2012.0101_0110.L4.CAC.CI.MAXIMUM_10day_new.tif
            ts_2011.0101_0110.L4.CAC.CI_TA.MAXIMUM_10day_new.tif
            ts_2011.0101_0110.L4.CAC.CIcyano.MAXIMUM_10day_new.tif
            ts_2011.0101_0110.L4.CAC.CI_merge.MAXIMUM_10day_new.tif
            ts_2011_0101_0110_L4_CAC_CI_TA_MAXIMUM_10day_new
            ts_2011_0101_0110_L4_CAC_CI_merge_TA_MAXIMUM_10day_new
            
        Returns extract components as a tuple (year, mm, dd, hhmm, area_name, prod_name, prod_ver). Values will
        be assigned None if unable to parse filename.
        '''
        L3_fn_parser_def = self.fn_parser_def
        
        dirname, fn_base = os.path.split(fn)
        year = None
        mm = None
        dd = None
        hhmm = None
        area_name = None
        prod_name = None
        prod_ver = None

        #split filename based on RS Tools default L4 filename delimiter (varies depending on if in GDB or folder)
        delim = '.' if L3_fn_parser_def['delim'] in fn_base else self.ws_delimiter
        fn_parts = fn_base.split(delim)
        
        if self.isNASA_fn(fn):
            s_time = fn_parts[0][1:8]
            #if multiday composite get etime else set etime to stime
            e_time = s_time if len(fn_parts[0]) == 8 else fn_parts[0][8:15]

            #return middle of date range
            s_dt = datetime.strptime(s_time, '%Y%j')
            e_dt = datetime.strptime(e_time, '%Y%j') + timedelta(days=1)
            dt = s_dt + (e_dt - s_dt)/2
            year = dt.strftime('%Y')
            mm = dt.strftime('%m')
            dd = dt.strftime('%d')
            hhmm = '0000'

            prod_name = 'unk'
            for nasa_pn in ['_CI_noncyano_', '_CI_cyano_', '_CI_']:
                if fn_parts[1].find(nasa_pn) >= 0:
                    prod_name = nasa_pn.replace('_', '')
                    break
                                   
            #area is assigned the tile ID (row_col) 
            subparts = fn_parts[1].split('_')
            area_name = '%s_%s' % (subparts[-2], subparts[-1]) if subparts[-3] == '300m' else 'unk'
            
            #if self.debug: self.log_file.write ('Info - nasa fn: prod=%s**area_name=%s**delim=%s**fn_parts[1]=%s**subparts=%s\n' % (prod_name, area_name, delim, fn_parts[1], subparts))
            
        elif self.isS2_MSI_fn(fn):
            fn_parts = fn_base.split('_')
            
            s_time = fn_parts[2][0:8]
            #not expecting multiday composite so set etime to stime
            e_time = s_time 

            #return middle of date range
            s_dt = datetime.strptime(s_time, '%Y%m%d')
            e_dt = datetime.strptime(e_time, '%Y%m%d') + timedelta(days=1)
            dt = s_dt + (e_dt - s_dt)/2
            year = dt.strftime('%Y')
            mm = dt.strftime('%m')
            dd = dt.strftime('%d')
            hhmm = '0000'            
                        
            prod_name = 'unk'
            #currently supported products available in run_gpt_s2.py
            for hab_pn in ['_MCI', '_gil_chl', '_truecolor']:
                if fn_base.find(hab_pn) >= 0:
                    prod_name = hab_pn[1:]
                    break
                                   
            #area is assigned the tile ID (row_col) 
            area_name = fn_parts[5] if len(fn_parts) >=5 else 'unk'
            
        elif len(fn_parts)<3 or fn_parts[2] != 'L4':
            #assuming L3 type filename
            #update fn_parts based on defined L3 delimiter
            fn_parts = fn_base.split(L3_fn_parser_def['delim'])
            # if self.debug: self.log_file.write ('len(fn_parts)=%s**L3_fn_parser_def["delim"]=%s**delim=%s\n' % (len(fn_parts), L3_fn_parser_def['delim'], delim))
           
            offset = 1 if fn_parts[0] == 'sentinel-3' else 0
            
            year = self.extract_fn_part(fn_parts, L3_fn_parser_def['year'])
            mm = self.extract_fn_part(fn_parts, L3_fn_parser_def['mm'])
            dd = self.extract_fn_part(fn_parts, L3_fn_parser_def['dd'])
            hhmm = self.extract_fn_part(fn_parts, L3_fn_parser_def['hhmm'])
            try:
                datetime.strptime( '%s%s%s %s' % (year, mm, dd, hhmm), '%Y%m%d %H%M')
            except ValueError:
                if L3_fn_parser_def['name'] == 'RSTools_default':
                    arcpy.AddError('Unable to parse filename "%s". It appears it is not an RSTools default L3 filename.' % fn_base)
                else:
                    arcpy.AddError('Unable to parse filename "%s". It appears the filename does not match your custom L3 filename format defined using the "Define filename parsing format" tool.' % fn_base)
                raise filenameFormatError(fn)
                
            area_name = self.extract_fn_part(fn_parts, L3_fn_parser_def['area_name'], offset)
            if area_name is None:
                area_name = 'unk'
            
            prod_name = self.extract_fn_part(fn_parts, L3_fn_parser_def['prod_name'],offset)
            if prod_name is None or prod_name == 'tif':
                prod_name = 'unk'

            ver_info = self.extract_fn_part(fn_parts, L3_fn_parser_def['ver_info'], offset)
            ver_parts = ver_info.split('_') if ver_info is not None else []
            prod_ver = None if len(ver_parts) < 3 else ver_parts[2]

        elif fn_parts[2] == 'L4':
            year_str = fn_parts[0].split('_')[1] if '_' in fn_parts[0] else fn_parts[0]
            (s_year, e_year) = year_str.split('-') if '-' in year_str else (year_str, year_str)
            
            s_time, e_time = fn_parts[1].split('_')
            s_mm = s_time[0:2]
            s_dd = s_time[2:4]
            e_mm = e_time[0:2]
            e_dd = e_time[2:4]

            #return middle of date range
            s_dt = datetime(int(s_year), int(s_mm), int(s_dd))
            e_dt = datetime(int(e_year), int(e_mm), int(e_dd)) + timedelta(days=1)
            dt = s_dt + (e_dt - s_dt)/2
            year = dt.strftime('%Y')
            mm = dt.strftime('%m')
            dd = dt.strftime('%d')
            hhmm = '0000'

            area_name = fn_parts[3]
            prod_name = fn_parts[4]

        return (year, mm, dd, hhmm, area_name, prod_name, prod_ver)            
        
        
    def extract_fn_part(self, fn_parts, part_desc, offset=0):
        '''
        Extract the part of a filename from of list of parts based on part_desc. "part_desc" is
        a list of 0, 1 or 3 items defining filename part to extract.
        
            part_desc[0] = index within the fn_parts list (required)
            part_desc[1] = starting index position within the fn_part string. if no value the fn_part will not be subset. (optional)
            part_desc[2] = ending index position within the fn_part string. if no value the fn_part will not be subset. (optional)
        
        Returns extracted fn_part or None if invalid definition for fn_parts or part_desc = []
        '''
        fn_part = None
        
        if len(part_desc) > 0:
            loc = part_desc[0] + offset
            spos, epos = (part_desc[1], part_desc[2]) if len(part_desc) == 3 else (None, None)
            
            if len(fn_parts) >= loc + 1:
                fn_part = fn_parts[loc] if spos is None else fn_parts[loc][spos:epos]
            # else:
                # fn_part = None
            
        return fn_part
        
    def validate_data(self, val, nodata_val=None):
        '''
        Validate data based on user specified min/max valid data ranges.

        Returns val in within range.
        Returns '' if val outside range or None.
        '''

        if val is None: return ''

        if nodata_val and val == nodata_val: return ''

        valid_val = val if self.min_valid_val is None or (self.min_valid_val is not None and val >= self.min_valid_val) else ''
        valid_val = valid_val if self.max_valid_val is None or (self.max_valid_val is not None and val <= self.max_valid_val) else ''

        return valid_val

    def raster_pixel_type (self, raster_fn):
        '''
        Return pixel type for first band of raster_fn
        '''
        try:
            desc = arcpy.Describe('%s\\Band_1' % raster_fn)
        except OSError:
            #not all single bands raster can be referenced as above
            desc = arcpy.Describe(raster_fn)
        if self.debug: self.log_file.write ('raster pixel type=%s.\n' % desc.pixelType)
        return desc.pixelType

    def validate_nodata(self, nodata_value, raster_fn):
        '''
        Verify the nodata value is within range the minimum and maximum values
        supported by the raster's pixel type.

        If outside range, return False and print warning.
        '''

        pixel_type = self.raster_pixel_type(raster_fn)
        pixel_size = int(pixel_type[1:])

        if pixel_type[0] == 'U':
            min_val = 0
            max_val = 2**pixel_size-1
            valid = True if (min_val <= nodata_value <= max_val) else False
        elif pixel_type[0] == 'S':
            min_val = -(2**pixel_size/2+1)
            max_val = (2**pixel_size)/2

            valid = True if (min_val <= nodata_value <= max_val) else False
        else:
            #we'll assume it's OK for floating point rasters
            valid = True

        if not valid:
            arcpy.AddWarning('The specified "Nodata value" of %s is outside the range of values supported by the raster pixel type. For the input files selected the "Nodata value" must be between %s and %s, inclusive.' % (nodata_value, min_val, max_val))

        return valid

    def can_unscale(self, fn):
        '''
        Return True if known product type and version and not a "real" (i.e. unscaled)
        product otherwise False
        '''
        
        (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(fn)

        if (prod_name.lower() in self.known_prods and 
            prod_ver in ["1", "2", "3"] and 
            f'{prod_name}.real.' not in fn):
            return True
        else:
            return False

    def get_unscaled_val(self, val, prod_name, prod_ver=None):
        '''
        For supported products unscaled value by applying unscaling formula

        Return unscaled value or "" if product not supported.
        '''
        
        if val and prod_name.lower() in self.known_prods:
            # if self.debug: self.log_file.write ('unscale - val=%s prod_name=%s\n' % (val, prod_name))
            if prod_ver == "1":
                return 10**(int(val)/100. - 4)
            elif prod_ver in ["2", "3"]:
                return 10**(3.0/250.0*int(val) - 4.2)
        return ''
    
    def copy_raster(self, in_raster, out_fn, raster_pixel_type, nodata_value):
        '''
        Copy raster setting the pixel type as specified by the arcpy raster pixel type.
        '''
        map_pixel_type = {'U1': "1_BIT", 
                          'U2': "2_BIT", 
                          'U4': "4_BIT", 
                          'U8': "8_BIT_UNSIGNED", 
                          'S8': "8_BIT_SIGNED", 
                          'U16': "16_BIT_UNSIGNED", 
                          'S16': "16_BIT_SIGNED", 
                          'U32': "32_BIT_UNSIGNED", 
                          'S32': "32_BIT_SIGNED", 
                          'F32': "32_BIT_FLOAT", 
                          'F64': "64_BIT", 
                          }
        out_pixel_type = map_pixel_type[raster_pixel_type]
        arcpy.CopyRaster_management(in_raster, out_fn, '', '', nodata_value, '', '', out_pixel_type, '', '')

    def create_masks(self, in_fns, mask_dir=arcpy.env.scratchFolder, mask_type='data'):
        '''
        For each file in in_fns list, create a new raster with value(s) outside
        the min/max valid data limits are set to the nodata_value.

        if mask_type == 'data'
             valid data values returned and invalid set to nodata
        if mask_type == 'binary'
            valid data pixels returned as 1 and invalid set to nodata

        Return list of masked filenames. Empty list returned if no min/max values
        given.
        '''
        masked_fns = []

        save_scratchWorkspace = arcpy.env.scratchWorkspace
        
        if self.debug: self.log_file.write ('create_masks: arcpy.env.workspace = %s\n' % (arcpy.env.workspace))
        if self.debug: self.log_file.write ('create_masks: arcpy.env.scratchWorkspace = %s\n' % (arcpy.env.scratchWorkspace))
        if self.debug: self.log_file.write ('create_masks: mask_dir = %s\n' % (mask_dir))

        exp = self.get_valid_data_exp(self.min_valid_val, self.max_valid_val)
        
        if exp is None:
            #no masking required
            if self.debug: self.log_file.write ('create_masks: no masking required\n')
            return []
            
        arcpy.env.overwriteOutput = True

        for fn in in_fns:
            mask_fn_base, ext = os.path.splitext(os.path.split(fn)[1])
            if self.out_fn_suffix:
                mask_fn_base += self.out_fn_suffix

            mask_fn = self.valid_fc_name(mask_fn_base, mask_dir)
            if self.debug: self.log_file.write ('fn=%s\nmask_fn_base=%s\next=%s\n' % (fn, mask_fn_base, ext))
            if arcpy.Describe(mask_dir).workspaceType == 'FileSystem':
                mask_fn += ext
                
            arcpy.AddMessage('Creating mask %s\n' % (mask_fn))
            if self.debug: self.log_file.write('maskdir.workspaceType=%s\n' % arcpy.Describe(mask_dir).workspaceType)
            if self.debug: self.log_file.write('create_masks: mask_fn = %s\n' % (mask_fn))

            in_raster = arcpy.Raster(fn)
            arcpy.env.scratchWorkspace = arcpy.env.scratchFolder if arcpy.Describe(os.path.dirname(fn)).workspaceType == 'FileSystem' else arcpy.env.scratchGDB
            
            if mask_type == 'data':
                #valid data values returned and invalid set to nodata
                masked_raster = Con(IsNull(in_raster), self.nodata_value, Con(in_raster, in_raster, self.nodata_value, exp))
            elif mask_type == 'binary':
                #valid data pixels returned as 1 and invalid set to nodata
                masked_raster = Con(IsNull(in_raster), self.nodata_value, Con(in_raster, 1, self.nodata_value, exp))

            # Con() appears to create raster pixel type F64. If the input raster is an integer and "median"
            #   statistic selected we get an error, so we force the pixel type of the masked raster to 
            #   match the input raster.
            #map the arcpy raster pixel type to the value expected by CopyRaster
            map_pixel_type = {'U1': "1_BIT", 
                              'U2': "2_BIT", 
                              'U4': "4_BIT", 
                              'U8': "8_BIT_UNSIGNED", 
                              'S8': "8_BIT_SIGNED", 
                              'U16': "16_BIT_UNSIGNED", 
                              'S16': "16_BIT_SIGNED", 
                              'U32': "32_BIT_UNSIGNED", 
                              'S32': "32_BIT_SIGNED", 
                              'F32': "32_BIT_FLOAT", 
                              'F64': "64_BIT", 
                              }
            out_pixel_type = map_pixel_type[in_raster.pixelType]
            arcpy.CopyRaster_management(masked_raster, mask_fn, '', '', self.nodata_value, '', '', out_pixel_type, '', '')

            if self.debug: self.log_file.write('masked_raster.isInt=%s pix_type=%s out_pixel_type=%s\n' % (masked_raster.isInteger, masked_raster.pixelType, out_pixel_type)) 
                
            #add color table for original file for integer type raster only
            if mask_type == 'data' and in_raster.isInteger:
                try:
                    arcpy.AddColormap_management(mask_fn, fn, "#")
                except arcpy.ExecuteError:
                    msgs = arcpy.GetMessages(2)

                    if self.log_file: self.log_file.write('Adding colormap failed: assuming the template file does not have one...continuing processing.')
                    if self.log_file: self.log_file.write('Reported arcpy exception: %s' % msgs)
                    pass

            if self.nodata_value is not None: arcpy.SetRasterProperties_management(mask_fn, '#', '#', '#', [[1, self.nodata_value]])

            masked_fns.append(mask_fn)

            del masked_raster
        
        arcpy.env.scratchWorkspace = save_scratchWorkspace
        
        return masked_fns

    def get_valid_data_exp(self, min_val, max_val):
        '''
        Return conditional expression as a string built from list min/max valid data values.
        '''
        exp_list = []
        if min_val is not None:
            exp_list.append('VALUE >= %s' % min_val)
        if max_val is not None:
            exp_list.append('VALUE <= %s' % max_val)

        exp = '(' + ' AND '.join(exp_list) + ')' if len(exp_list) > 0 else None

        if self.debug: self.log_file.write ('mask_valid_data: exp = %s\n' % (exp))

        return exp

    def valid_fc_name(self, fc_basename, ws):
        '''
        Ensures a valid output feature class name for a workspace. We're using '__' as the substitute for 
        the '.' delimiter in the SAPS filenames.
        
        Returns validates feature class prefixed with workspace.
        '''
        val_fc_basename = fc_basename.replace('.', self.ws_delimiter)
        val_fc_basename = arcpy.ValidateTableName(val_fc_basename, ws)
        return os.path.join(ws, val_fc_basename)

    def match_up_file_filter(self, in_files, fc_fn):
        '''
        Return a list of files from the in_files where the date
        falls within the matchup period of any feature within the 
        feature class.
        '''
        in_files_filtered = []

        flds = [self.match_date_fldname]

        for fn in in_files:
            (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(fn)
            fn_date = datetime.strptime( '%s%s%s %s' % (year, mm, dd, hhmm), '%Y%m%d %H%M')
            
            rows = arcpy.SearchCursor(fc_fn, "", "",  ';'.join(flds), "")

            for row in rows:
                pt_date = row.getValue(self.match_date_fldname)
            
                if pt_date is None:
                    arcpy.AddWarning('Field "%s" in "%s" feature class contains a NULL value. Row skipped.' % (self.match_date_fldname, fc_fn))
                    continue

                match = self.test_matchup(fn_date, pt_date)
                if match:
                    in_files_filtered.append(fn)
                    break
        del rows

        return in_files_filtered

    def test_matchup(self, fn_date, pt_date):
        '''
        Perform test to verify feature and raster dates within defined time window.

        Returns True if match
        '''
        if 'hour' in self.match_units:
            delta = timedelta(hours=self.match_time)

            # if self.debug: self.log_file.write('in test_matchup (hours) (NOTE using date & time portion):\nfn_date=%s*fc_date=%s**match=%s\n' % (fn_date, pt_date, pt_date-delta <= fn_date <= pt_date+delta))

            return pt_date-delta <= fn_date <= pt_date+delta
        else:
            delta = timedelta(days=self.match_time)

            # if self.debug: self.log_file.write('in test_matchup (days) (NOTE only using date() portion):\nfn_date=%s*fc_date=%s**match=%s\n' % (fn_date, pt_date, pt_date.date()-delta <= fn_date.date() <= pt_date.date()+delta))

            return pt_date.date()-delta <= fn_date.date() <= pt_date.date()+delta
            
    def clean_up(self):
        pass

    def report_error(self, logonly=False):
        '''
        '''
        # Get the traceback object
        extype, exval, extr = sys.exc_info()
        exception_list = traceback.format_exception(extype, exval, extr)

        # Concatenate information together concerning the error into a message string
        pymsg = "PYTHON ERRORS:\nTraceback info:\n" + '\n'.join(exception_list)
        msgs = "ArcPy ERRORS:\n" + arcpy.GetMessages(2) + "\n"

        # Return python error messages for use in script tool or Python Window
        if not logonly:
            arcpy.AddError(pymsg)
            arcpy.AddError(msgs)

        if self.log_file:
            self.log_file.write(pymsg + "\n")
            self.log_file.write(msgs)

####
##  Python Toolbox setup
####
class Toolbox(object):
    def __init__(self):
        self.label = 'RS Tools'
        self.alias = ''
        self.tools = [CreateTimeSeries, RasterStatsByPolygon, MaskNoData, PixelExtractByPoint, PixelExtractByPolygon, SwapColormap]

# Tool implementation code
class CreateTimeSeries(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""

            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            # #since the "optional" / "required" flag is read only we define parameter
            # #   as required even though it's not always "required" as in this situation
            # #   so we're just assigning a date; the value will be ignored in processing
            # if not self.params[method_pnum+1].enabled and not self.params[method_pnum+1].value:
                # self.params[method_pnum+1].value = arcpy.env.workspace 

            # if not self.params[method_pnum+2].enabled and not self.params[method_pnum+2].value:
                # self.params[method_pnum+2].value = '*.tif'

            # if not self.params[method_pnum+4].enabled and not self.params[method_pnum+4].value:
                # self.params[method_pnum+4].value = 'dummy.txt'
            
            comptype_pnum = 6
            self.params[comptype_pnum+1].enabled = (self.params[comptype_pnum].value.lower() == 'timerange' or
                                      self.params[comptype_pnum].value.lower() == 'seasonal')
            self.params[comptype_pnum+2].enabled = (self.params[comptype_pnum].value.lower() == 'timerange')
            self.params[comptype_pnum+3].enabled = (self.params[comptype_pnum].value.lower() == 'timerange')
            #combine years only enabled for specific composite types
            self.params[comptype_pnum+4].enabled = (self.params[comptype_pnum].value.lower() == 'monthly' or
                                    (self.params[comptype_pnum].value.lower() == 'timerange' and self.params[comptype_pnum+3].value) or
                                    self.params[comptype_pnum].value.lower() == 'seasonal')
            #reset to false if not enabled
            if not self.params[comptype_pnum+4].enabled:
                self.params[comptype_pnum+4].value = False

            copylm_pnum = 11
            self.params[copylm_pnum+1].enabled = self.params[copylm_pnum].value

            prefix_pnum = 15
            if self.params[prefix_pnum].value:
                if self.params[prefix_pnum].value[-1] != '_':
                    self.params[prefix_pnum].value += '_'

            suffix_pnum = 16
            if self.params[suffix_pnum].value:
                if self.params[suffix_pnum].value[0] != '_':
                    self.params[suffix_pnum].value = '_' + self.params[suffix_pnum].value

            if (not self.params[comptype_pnum].hasBeenValidated or not self.params[comptype_pnum+2].hasBeenValidated):
                if self.params[comptype_pnum].value.lower() == 'timerange':
                    self.params[suffix_pnum].value = '_%sday' % self.params[comptype_pnum+2].value
                else:
                    self.params[suffix_pnum].value = '_' + self.params[comptype_pnum].value.lower()

            if not self.params[comptype_pnum+1].enabled and not self.params[comptype_pnum+1].value:
                #since the "optional" / "required" flag is read only we define parameter
                #   as required even though it's not always "required" as in this situation
                #   so we're just assigning a date; the value will be ignored in processing
                self.params[comptype_pnum+1].value = '1-1-2015'

            if not self.params[comptype_pnum+2].enabled and not self.params[comptype_pnum+2].value:
                #since the "optional" / "required" flag is read only we define parameter
                #   as required even though it's not always "required" as in this situation
                #   so we're just assigning a date; the value will be ignored in processing
                self.params[comptype_pnum+2].value = '10'

            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            prefix_pnum = 15
            if self.params[prefix_pnum].value:
                if self.params[prefix_pnum].value[0].isdigit():
                    self.params[prefix_pnum].setErrorMessage('Filename prefix cannot begin with a number')
                elif '_' in self.params[prefix_pnum].value[:-1]:
                    self.params[prefix_pnum].setErrorMessage('Filename prefix can only include a single "_" at the end of the string.')
                else:
                    self.params[prefix_pnum].clearMessage()

            # if self.params[13].value:
                # if '_' in self.params[13].value[1:]:
                    # self.params[13].setErrorMessage('Filename suffix can only include a single "_" at the start of the string.')
                # else:
                    # self.params[13].clearMessage()
            comptype_pnum = 6
            if self.params[comptype_pnum+2].enabled and self.params[comptype_pnum+2].value <= 0:
                self.params[comptype_pnum+2].setErrorMessage('Duration must be greater than zero.')

            return

    def __init__(self):
        super(CreateTimeSeries, self ).__init__()
        self.label = 'Create time series composites'
        self.description = 'Create time series by compositing list of input files.'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]

        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive_search
        param3 = arcpy.Parameter()
        param3.name = 'Recursive_search'
        param3.displayName = 'Recursive search'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'

        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'

        # Which_band
        param4 = arcpy.Parameter()
        param4.name = 'Which_band'
        param4.displayName = 'Which band'
        param4.parameterType = 'Required'
        param4.direction = 'Input'
        param4.datatype = 'Long'
        param4.value = '1'

        # Composite_Type
        param5 = arcpy.Parameter()
        param5.name = 'Composite_Type'
        param5.displayName = 'Composite Type'
        param5.parameterType = 'Required'
        param5.direction = 'Input'
        param5.datatype = 'String'
        param5.value = 'Monthly'
        param5.filter.list = ['Monthly', 'Yearly', 'Seasonal', 'TimeRange', 'All']

        # Start_date
        param6 = arcpy.Parameter()
        param6.name = 'Start_date'
        param6.displayName = 'Start date'
        param6.parameterType = 'Required'
        param6.direction = 'Input'
        param6.datatype = 'Date'
        param6.value = '1-1-2015'

        # Period_duration__days_
        param7 = arcpy.Parameter()
        param7.name = 'Period_duration__days_'
        param7.displayName = 'Period duration (days)'
        param7.parameterType = 'Required'
        param7.direction = 'Input'
        param7.datatype = 'Long'
        param7.value = '10'

        # Do_not_span_months
        param8 = arcpy.Parameter()
        param8.name = 'Do_not_span_months'
        param8.displayName = 'Do not span months'
        param8.parameterType = 'Required'
        param8.direction = 'Input'
        param8.datatype = 'Boolean'
        param8.value = 'false'

        # Combine years
        param_comb_years = arcpy.Parameter()
        param_comb_years.name = 'Combine_Years'
        param_comb_years.displayName = 'Combine Years'
        param_comb_years.parameterType = 'Required'
        param_comb_years.direction = 'Input'
        param_comb_years.datatype = 'Boolean'
        param_comb_years.value = 'false'

        # Copy_landmask
        param9 = arcpy.Parameter()
        param9.name = 'Copy_landmask'
        param9.displayName = 'Copy landmask'
        param9.parameterType = 'Required'
        param9.direction = 'Input'
        param9.datatype = 'Boolean'
        param9.value = 'true'

        # Landmask_value
        param10 = arcpy.Parameter()
        param10.name = 'Landmask_values'
        param10.displayName = 'Landmask value'
        param10.parameterType = 'Required'
        param10.direction = 'Input'
        param10.datatype = 'Double'
        param10.value = '252'

        # Statistic
        param11 = arcpy.Parameter()
        param11.name = 'Statistic'
        param11.displayName = 'Statistic'
        param11.parameterType = 'Required'
        param11.direction = 'Input'
        param11.datatype = 'String'
        param11.value = 'MEDIAN'
        param11.filter.list = ['MEAN', 'MEDIAN', '10PERCENTILE', '25PERCENTILE', '75PERCENTILE', '90PERCENTILE', 'MINIMUM', 'MAXIMUM', 'STD']

        # Create_count
        param12 = arcpy.Parameter()
        param12.name = 'Create_count'
        param12.displayName = 'Create count'
        param12.parameterType = 'Required'
        param12.direction = 'Input'
        param12.datatype = 'Boolean'
        param12.value = 'false'

        # Output_filename_prefix
        param13 = arcpy.Parameter()
        param13.name = 'Output_filename_prefix'
        param13.displayName = 'Output filename prefix'
        param13.parameterType = 'Required'
        param13.direction = 'Input'
        param13.datatype = 'String'
        param13.value = 'ts_'

        # Output_filename_suffix
        param14 = arcpy.Parameter()
        param14.name = 'Output_filename_suffix'
        param14.displayName = 'Output filename suffix'
        param14.parameterType = 'Optional'
        param14.direction = 'Input'
        param14.datatype = 'String'

        # Output_workspace
        param15 = arcpy.Parameter()
        param15.name = 'Output_workspace'
        param15.displayName = 'Output workspace'
        param15.parameterType = 'Required'
        param15.direction = 'Input'
        param15.datatype = ('Workspace')

        # Minimum valid_data value
        param16 = arcpy.Parameter()
        param16.name = 'Minimum_valid_data_value'
        param16.displayName = 'Minimum valid data value'
        param16.parameterType = 'Optional'
        param16.direction = 'Input'
        param16.datatype = 'Double'
        param16.value = '1'

        # Maximum valid_data value
        param17 = arcpy.Parameter()
        param17.name = 'Maximum_valid_data_value'
        param17.displayName = 'Maximum valid data value'
        param17.parameterType = 'Optional'
        param17.direction = 'Input'
        param17.datatype = 'Double'
        param17.value = '250'

        # Nodata_value
        param18 = arcpy.Parameter()
        param18.name = 'Nodata_value'
        param18.displayName = 'Nodata value'
        param18.parameterType = 'Required'
        param18.direction = 'Input'
        param18.datatype = 'Double'
        param18.value = '255'

        return [param0, param1, param2, param3, param1b, param4, param5, param6, param7, param8, param_comb_years, param9, param10, param11, param12, param13, param14, param15, param16, param17, param18]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:

            # ##DEBUGGING
#           h = hpy()
            h = None
            self.h = h

            num_args = len(parameters)
            tool_name = 'time_series'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.band_num = parameters[5].value

            # self.in_ws = parameters[0].valueAsText
            # self.in_filter = parameters[1].valueAsText
            # self.band_num = parameters[2].value
            # self.recursive = parameters[3].value
            self.comp_type = parameters[6].valueAsText.lower()
            self.start_date = parameters[7].value
            self.duration = parameters[8].value
            self.do_not_span_months = parameters[9].value
            self.combine_years = parameters[10].value
            self.copy_landmask = parameters[11].value
            self.landmask_value = parameters[12].value
            self.stat_name = parameters[13].valueAsText.upper()
            self.create_count = parameters[14].value
            self.out_fn_prefix = self.validate_fn_prefix(parameters[15].valueAsText)
            self.out_fn_suffix = self.validate_fn_suffix(parameters[16].valueAsText)
            self.out_ws = parameters[17].valueAsText
            self.min_valid_val = parameters[18].value
            self.max_valid_val = parameters[19].value
            self.nodata_value = parameters[20].value

            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_ext = self.get_out_ext(self.out_ws_type)

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\nselection_method=%s\nin_ws=%s\nin_filter=%s\nband_num=%s\nrecursive=%s\nin_filelist_fn=%s\n' % (self.input_method, self.in_ws, self.in_filter, self.band_num, self.recursive, self.in_filelist_fn))
                self.log_file.write ('comp_type=%s\ncreate_count=%s\nstart_date=%s\nduration=%s\nstatistic=%s\n' % (self.comp_type, self.create_count, self.start_date, self.duration, self.stat_name))
                self.log_file.write ('do_not_span_months=%s\n' % (self.do_not_span_months))
                self.log_file.write ('combine years=%s\n' % (self.combine_years))
                self.log_file.write ('copy_landmask=%s\n' % (self.copy_landmask))
                self.log_file.write ('landmask_value=%s\n' % (self.landmask_value))
                self.log_file.write ('min_valid_val=%s\n' % (self.min_valid_val))
                self.log_file.write ('max_valid_val=%s\n' % (self.max_valid_val))
                self.log_file.write ('out_fn_prefix=%s\nout_fn_suffix=%s\nout_ws=%s\nout_ext=%s\nnodata_val=%s\n' % (self.out_fn_prefix, self.out_fn_suffix, self.out_ws, self.out_ext, self.nodata_value))
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)
                
                if h:
                    self.log_file.write('1heap=%s\n' % h.heap())
                    h.setrelheap()

            self.parameters = parameters

            self.get_arc_lic('Spatial')

            in_files = self.get_files()

            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            valid = self.validate_nodata(self.nodata_value, in_files[0])
            if not valid:
                arcpy.AddError('Invalid nodata value...exiting.')
                return

            arcpy.env.overwriteOutput = True

            if h and self.debug: self.log_file.write('2heap=%s\n' % h.heap())
            if h and self.debug: self.log_file.write('2heap.bytype=%s\n' % h.heap().bytype)
            if h and self.debug: self.log_file.write('2heap.byrcs=%s\n' % h.heap().byrcs)
                
            self.composite_files(in_files)
            if h and self.debug: self.log_file.write('3heap=%s\n' % h.heap())

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)
            arcpy.env.overwriteOutput = self.orig_overwriteoutput
            del in_files


    def composite_files(self, in_files):
        '''
        '''

        try:
            fn_dict = self.fn_info_by_date(in_files)
            if self.h and self.debug: self.log_file.write('4heap.bytype=%s\n' % self.h.heap().bytype)
            if self.h and self.debug: self.log_file.write('4heap.byrcs=%s\n' % self.h.heap().byrcs)

            if self.debug: self.log_file.write ('%s valid filenames parsed.\n' % len(fn_dict))
            if self.debug: self.log_file.write ('file dates: %s\n' % sorted(fn_dict))

            if not fn_dict:
                arcpy.AddWarning('No files found to process...exiting')
                return

            if self.comp_type == 'timerange':
                if self.duration <= 0:
                    arcpy.AddError('Duration must be greater than zero.')
                    return

                if not self.start_date:
                    arcpy.AddError('Start date required for "timerange" comp_type.')
                    return

                if self.start_date > (sorted(fn_dict))[-1]:
                    arcpy.AddError('No raster files found after the start date (%s)' % self.start_date)
                    return

            if self.comp_type == 'seasonal':
                if not self.start_date:
                    arcpy.AddError('Start date required for "seasonal" comp_type.')
                    return

                if self.start_date > (sorted(fn_dict))[-1]:
                    arcpy.AddError('No raster files found after the start date (%s)' % self.start_date)
                    return

            if self.band_num <= 0:
                arcpy.AddError('Invalid band_num (%s). Value must be > than zero but <= number of bands with input files.' % self.band_num)
                return

            if self.comp_type == 'all':
                self.make_composite_all_fns(fn_dict)
            else:
                comp_fns_dict = self.fns_by_period(fn_dict)
                self.make_composites(comp_fns_dict)

            del fn_dict

            arcpy.AddMessage('done compositing.')

        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)

        except:
            self.report_error()

    def fns_by_period(self, fn_dict):
        '''
        Step through each composite time period to identify files within period.

        Return a dictionary containing filenames organized by start of period and year.
        '''
        try:
            comp_fns = []
            area_name = None
            prod_name = None
            comp_fns_dict = {}

            sorted_dts = sorted(fn_dict)

            if self.comp_type == 'monthly':
                start_dt = datetime(sorted_dts[0].year, sorted_dts[0].month, 1)
            elif self.comp_type == 'yearly':
                start_dt = datetime(sorted_dts[0].year, 1, 1)
            elif self.comp_type == 'all':
                start_dt = sorted_dts[0]
            else:
                # 'timerange' & 'seasonal' use user defined start date
                start_dt = self.start_date

            if self.comp_type == 'all':
                end_dt = sorted_dts[-1]
            else:
                end_dt = self.get_end_date(start_dt)

            arcpy.AddMessage('compositing: first file date=%s / last file date=%s\n' % (sorted_dts[0], sorted_dts[-1]))
            if self.debug: self.log_file.write ('compositing: first file date=%s / last file date=%s\n' % (sorted_dts[0], sorted_dts[-1]))
            last_file_dt = sorted_dts[-1]

            #note: binning is only considering the date component (time component ignored)
            for dt in sorted_dts:

                #skip files prior to start date
                if dt.date() < start_dt.date():
                    if self.debug: self.log_file.write ('skipping...file dt=%s prior to sdt=%s\n' % (dt.date(), start_dt.date()))
                    continue

                #current file outside current time range
                if (not (start_dt.date() <= dt.date() <= end_dt.date())):
                    if self.debug: self.log_file.write ('identifying files in range: Y-m-d = %s-%s-%s to %s\n' % (start_dt.year, start_dt.month, start_dt.day,  end_dt.strftime('%Y-%m-%d')))
                    if self.debug: self.log_file.write ('identifying files in range: # files = %s\n' % (len(comp_fns)))

                    if comp_fns:
                        if start_dt.strftime('%m%d') not in comp_fns_dict:
                            comp_fns_dict[start_dt.strftime('%m%d')] = {}
                        comp_fns_dict[start_dt.strftime('%m%d')][start_dt.year] = comp_fns
                        comp_fns = []
                        area_name = None
                        prod_name = None

                    #move to next time window that includes current file date
                    while (dt.date() > end_dt.date()):
                        start_dt = end_dt  + timedelta(1)
                        end_dt = self.get_end_date(start_dt)

                        if self.debug:
                            if dt.date() > end_dt.date(): self.log_file.write ('No files with time range: sdt=%s to edt=%s\n' % (start_dt.date(), end_dt.date()))

                comp_fns.append(fn_dict[dt])

            #include remaining files
            if comp_fns:
                if start_dt.strftime('%m%d') not in comp_fns_dict:
                    comp_fns_dict[start_dt.strftime('%m%d')] = {}
                comp_fns_dict[start_dt.strftime('%m%d')][start_dt.year] = comp_fns

            return comp_fns_dict

        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)

        except:
            self.report_error()


    def make_composites(self, comp_fns_dict):
        '''
        Generate composites for all filenames in each time period.

        If combine_years then all years of filenames in a period will be combined in a single composite
        otherwise composites will be generated for each period and year combination.
        '''
        try:
            
            for start_mmdd in sorted(comp_fns_dict.keys()):

                if self.debug: self.log_file.write ('*compositing: period start = %s\n' % (start_mmdd))
                    
                if self.combine_years:
                    comp_fns = []
                    years = sorted(comp_fns_dict[start_mmdd].keys())
                    if self.debug: self.log_file.write ('*compositing: years = %s\n' % (years))

                    for year in years:
                        comp_fns.extend(comp_fns_dict[start_mmdd][year])

                    start_dt = datetime.strptime('%s%s' % (years[0], start_mmdd), '%Y%m%d')
                    end_dt = self.get_end_date(datetime.strptime('%s%s' % (years[-1], start_mmdd), '%Y%m%d'))
                    if self.debug: self.log_file.write ('end_dt=%s\n' % (end_dt))

                    prod_name, area_name, prod_ver = self.validate_comp_fns(comp_fns)
                    out_fn = self.composite_fn(area_name, prod_name, prod_ver, start_dt, end_dt)

                    if self.debug: self.log_file.write ('*compositing: Yrs-m-d = %s-%s-%s\n' % (str(years), start_dt.month, start_dt.day))
                    if self.debug: self.log_file.write ('*compositing: # files = %s\n' % (len(comp_fns)))
                    if self.debug: self.log_file.write ('*compositing: out_fn = %s\n' % (out_fn))
                    if self.debug: self.log_file.write ('*compositing: files = %s\n' % (comp_fns))

                    arcpy.AddMessage('Creating composite (combined years) %s' % out_fn)
                    self.composite(comp_fns, out_fn)

                else:

                    for year in sorted(comp_fns_dict[start_mmdd].keys()):
                        comp_fns = comp_fns_dict[start_mmdd][year]

                        start_dt = datetime.strptime('%s%s' % (year, start_mmdd), '%Y%m%d')
                        end_dt = self.get_end_date(datetime.strptime('%s%s' % (year, start_mmdd), '%Y%m%d'))
                        if self.debug: self.log_file.write ('end_dt=%s\n' % (end_dt))

                        prod_name, area_name, prod_ver = self.validate_comp_fns(comp_fns)
                        out_fn = self.composite_fn(area_name, prod_name, prod_ver, start_dt, end_dt)

                        if self.debug: self.log_file.write ('*compositing: Y-m-d = %s-%s-%s\n' % (start_dt.year, start_dt.month, start_dt.day))
                        if self.debug: self.log_file.write ('*compositing: # files = %s\n' % (len(comp_fns)))
                        if self.debug: self.log_file.write ('*compositing: out_fn = %s\n' % (out_fn))
                        if self.debug: self.log_file.write ('*compositing: files = %s\n' % (comp_fns))

                        arcpy.AddMessage('Creating composite %s' % out_fn)
                        self.composite(comp_fns, out_fn)
                    
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)

        except InvalidBandNum as e:
            pass

        except:
            self.report_error()

    def make_composite_all_fns(self, fn_dict):
        '''
        Generate composite for Composite Type "All" using all valid filenames in filename dictionary.
        '''
        try:
            sorted_dts = sorted(fn_dict)
            start_dt = sorted_dts[0]
            end_dt = sorted_dts[-1]
            arcpy.AddMessage('compositing: first file date=%s / last file date=%s\n' % (start_dt, end_dt))

            comp_fns = [fn_dict[dt] for dt in sorted_dts]
            prod_name, area_name, prod_ver = self.validate_comp_fns(comp_fns)
            out_fn = self.composite_fn(area_name, prod_name, prod_ver, start_dt, end_dt)

            arcpy.AddMessage('Creating composite (all files) %s' % out_fn)
            self.composite(comp_fns, out_fn)            

        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)

        except InvalidBandNum as e:
            pass

        except:
            self.report_error()

    def composite_fn(self, area_name, prod_name, prod_ver, start_dt, end_dt):
        '''
        Build composite filename with format:
            [<prefix>_]yyyy.<start mmdd>_<end mmdd>.L4.<stat_name>[_<suffix>][.<.ext>]

        Return filename
        '''
        # (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(in_fn)

        #does composite span years?
        if start_dt.year != end_dt.year:
            fn = '{0}{1:04d}-{10:04d}.{2:02d}{3:02d}_{4:02d}{5:02d}.L4.{6}.{7}.{8}{9}'.format(self.out_fn_prefix, start_dt.year, start_dt.month, start_dt.day, end_dt.month, end_dt.day, area_name, prod_name, self.stat_name, self.out_fn_suffix, end_dt.year)
        else:
            fn = '{0}{1:04d}.{2:02d}{3:02d}_{4:02d}{5:02d}.L4.{6}.{7}.{8}{9}'.format(self.out_fn_prefix, start_dt.year, start_dt.month, start_dt.day, end_dt.month, end_dt.day, area_name, prod_name, self.stat_name, self.out_fn_suffix)

        if prod_ver is not None:
            fn = fn.replace(f'.{prod_name}.', f'.{prod_name}.v{prod_ver}.')

        if self.out_ws_type != 'FileSystem':
            fn = fn.replace('.', self.ws_delimiter)

        if self.out_ext:
            fn += '.%s' % self.out_ext
        return fn

    def validate_comp_fns(self, comp_fns):
        '''
        Validate product and area names in list are consistent. Generate warning if inconsistency detected.

        Return tuple containing product name and area name.
        '''
        area_name = None
        prod_name = None
        prod_ver = None

        for fn in comp_fns:
            (year, mm, dd, hhmm, fn_area_name, fn_prod_name, fn_prod_ver) = self.parse_fn(fn)

            if not area_name:
                area_name = fn_area_name
            elif area_name != fn_area_name:
                arcpy.AddWarning('Combining multiple regions (%s and %s) into a single composite...processing will continue but verify this is actually what you want to do.' % (area_name, fn_area_name))

            if not prod_name:
                prod_name = fn_prod_name
            elif prod_name != fn_prod_name:
                arcpy.AddWarning('Combining multiple product types (%s and %s) into a single composite...processing will continue but verify this is actually what you want to do.' % (prod_name, fn_prod_name))
                
            if not prod_ver:
                prod_ver = fn_prod_ver
            elif prod_ver != fn_prod_ver:
                arcpy.AddWarning('Combining multiple product versions (%s and %s) into a single composite...processing will continue but verify this is actually what you want to do.' % (prod_ver, fn_prod_ver))

        return (prod_name, area_name, prod_ver)

    def composite(self, comp_fns, out_fn):
        '''
        Composite list of files to single raster.
        '''
        try:
            save_ws = arcpy.env.workspace
            band_idx = self.band_num - 1

            arcgis_ver_str = arcpy.GetInstallInfo()["Version"]
            arcgis_ver_parts = arcgis_ver_str.split('.')
            if self.debug: self.log_file.write ('Arcgis installinfo=%s\n' % (arcpy.GetInstallInfo()))

            tmp_stats_block_fns = []
            tmp_count_block_fns = []
            
            #only create raster objects once otherwise very slow for input files on network
            #   - on our network, reading files from network instead on local drive adds 20+ 
            #        secs per input raster
            in_rasters = []
            for i, fn in enumerate(comp_fns):
                
                in_rasters.append(arcpy.Raster(fn))
                
            # use first file to initialize values (all files are expected to be the same!!)
            band_count = in_rasters[0].bandCount
            if self.band_num > band_count: raise InvalidBandNum('%s,%s' % (self.band_num, band_count))

            if self.debug: self.log_file.write ('len(comp_fns)=%s, in_rasters[0].height=%s, in_rasters[0].width=%s\n' % (len(comp_fns), in_rasters[0].height, in_rasters[0].width))

            offset = 1 if band_count > 1 else 0

            #arcpy.NumPyArrayToRaster will respect env setting
            arcpy.env.overwriteOutput = True
            arcpy.env.outputCoordinateSystem = comp_fns[0]
            x_cellsize = in_rasters[0].meanCellWidth
            y_cellsize = in_rasters[0].meanCellHeight
            min_map_llx = in_rasters[0].extent.XMin
            min_map_lly = in_rasters[0].extent.YMin

            raster_width = in_rasters[0].width
            raster_height = in_rasters[0].height
            
            # Process files in blocks
            # Size of processing data block where memorysize = datatypeinbytes*len(comp_fns)*blocksize^2
            # manage memory usage based on number of files we're compositing
            maxsize = 100   #in MB
            for blocksize in [1024, 512, 256, 128]:
                if ((4*len(comp_fns)*blocksize**2)/(1024*1024) <= maxsize): break

            if self.debug: self.log_file.write ('Blksize=%s Memsize=%s MB\n' % (blocksize, (4*len(comp_fns)*blocksize**2)/(1024*1024)))

            block_no = 0
            for x in range(0, raster_width, blocksize):
                for y in range(0, raster_height, blocksize):

                    for i, fn in enumerate(comp_fns):
           
                        # Lower left coordinate of block (in map units)
                        map_llx = min_map_llx + x * x_cellsize
                        map_lly = min_map_lly + y * y_cellsize
                        # Upper right coordinate of block (in cells)
                        urx = min([x + blocksize, raster_width])
                        ury = min([y + blocksize, raster_height])
                        #   noting that (x, y) is the lower left coordinate (in cells)
           
                        # Extract data block
                        in_data = arcpy.RasterToNumPyArray(in_rasters[i], arcpy.Point(map_llx, map_lly), urx-x, ury-y)

                        if self.debug: self.log_file.write ('processing %s; file_num=%s block_no=%s\n' % (fn, i, block_no))

                        if i == 0:
                            if self.debug: self.log_file.write ('dtype=%s\n' % (in_data.dtype))
                            if self.debug: self.log_file.write ('len(comp_fns)=%s  in_data.shape[0+off]=%s  in_data.shape[1+off]=%s  offset=%s\n' % (len(comp_fns), in_data.shape[0+offset], in_data.shape[1+offset], offset))
                            all_data = ma.empty((len(comp_fns), in_data.shape[0+offset], in_data.shape[1+offset]), dtype=in_data.dtype)
                            if self.debug: self.log_file.write ('all_data.shape=' + repr(all_data.shape) + '\n')
                            self.indata_dtype = in_data.dtype
                        else:
                            #couple of consistency checks
                            if (in_rasters[i].bandCount != band_count):
                                if block_no == 0: arcpy.AddWarning('Skipping %s...Band count (%s) does not match first file (%s).' % (fn, in_rasters[i].bandCount, band_count))
                                continue

                            if (not(isclose(x_cellsize, in_rasters[i].meanCellWidth)) or
                                not(isclose(y_cellsize, in_rasters[i].meanCellHeight))):
                                if block_no == 0: arcpy.AddWarning('Skipping %s...Cell size (%s,%s) does not match first file (%s,%s).' % (fn, in_rasters[i].meanCellWidth, in_rasters[i].meanCellHeight, x_cellsize, y_cellsize))
                                continue                              

                        if self.debug: self.log_file.write ('in_data.shape=' + repr(in_data.shape) + '\n')
                        if self.debug: self.log_file.write ('in_data=%s\ntype(in_data)=%s\n' % (in_data, type(in_data)))

                        if band_count > 1:
                            in_data = in_data[band_idx]

                        #mask nan or inf
                        in_data = ma.masked_invalid(in_data)

                        #mask data outside the user defined range for valid data value(s)
                        if self.min_valid_val is not None:
                            in_data = ma.masked_less(in_data, self.min_valid_val)

                        if self.max_valid_val is not None:
                            in_data = ma.masked_greater(in_data, self.max_valid_val)

                        in_data = in_data.reshape(1, in_data.shape[0], in_data.shape[1])

                        all_data[i] = in_data[0]
                                   
                    if self.stat_name == 'MEAN':
                        stats_data = all_data.mean(axis=0, dtype=np.float32)
                    elif self.stat_name == 'MEDIAN':                      
                        stats_data = ma.median(all_data, axis=0, overwrite_input=False)
                    elif self.stat_name == 'STD':
                        stats_data = all_data.std(axis=0, dtype=np.float32)
                    elif self.stat_name == 'MINIMUM':
                        stats_data = all_data.min(axis=0)
                    elif self.stat_name == 'MAXIMUM':
                        stats_data = all_data.max(axis=0)
                    elif 'PERCENTILE' in self.stat_name:
                        which_percentile = int(self.stat_name[0:2])
                        ## np.nanpercentile not available until numpy v1.9.0 & doesn't work on integer array
                        # stats_data = np.nanpercentile(all_data.filled(np.nan), which_percentile, axis=0)

                        # very slow
                        stats_data = ma.apply_along_axis(mquantiles, 0, all_data, prob=[which_percentile/100.], alphap=0, betap=1)
                                              
                    if self.debug: self.log_file.write ('stats_data.dtype=%s\n' % (stats_data.dtype))

                    out_dtype = in_data.dtype
                    tmp_fn = self.output_stats(stats_data, block_no, out_fn, self.out_ws, out_dtype, x_cellsize, y_cellsize, map_llx, map_lly, self.nodata_value)
                    tmp_stats_block_fns.append(tmp_fn)
           
                    if self.create_count:
                        stats_data = all_data.count(axis=0)
                        out_count_fn = out_fn.replace(self.stat_name, 'COUNT')
                        tmp_fn = self.output_stats(stats_data, block_no, out_count_fn, self.out_ws, np.int16, x_cellsize, y_cellsize, map_llx, map_lly, self.nodata_value)
                        tmp_count_block_fns.append(tmp_fn)

                    block_no += 1

            del in_data
            del in_rasters
            del all_data
            del stats_data

            self.mosaic_files(tmp_stats_block_fns, out_fn, self.out_ws, self.nodata_value, comp_fns[0])

            #can only propogate SAPS metadata for tif output 
            #   (& source NOAA/NCCOS products)
            if arcpy.Describe(self.out_ws).workspaceType == 'FileSystem':            
                self.add_saps_metadata(os.path.join(self.out_ws, out_fn), comp_fns)
            
            if self.create_count: self.mosaic_files(tmp_count_block_fns, out_fn.replace(self.stat_name, 'COUNT'), self.out_ws, self.nodata_value, None)

        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)

        except InvalidBandNum as e:
            arcpy.AddError('Invalid band_num (%s). Value must be <= number of bands with input files (%s).' % (e.band_num, e.band_count))
            raise
        except:
            self.report_error()

        finally:
            arcpy.env.workspace = self.out_ws
            # Remove temporary files
            for fn in tmp_stats_block_fns + tmp_count_block_fns:
                if self.debug: self.log_file.write ('Mosaic created...deleting temporary files\n')
                if arcpy.Exists(fn):
                    arcpy.Delete_management(fn)

            arcpy.env.workspace = save_ws
            arcpy.env.overwriteOutput = self.orig_overwriteoutput

    def mosaic_files(self, fns, out_fn, out_ws, nodata_value, src_tif):
        '''
        Mosaic all files in fns and then remove files in fns.
        '''
        # Mosaic temporary files
        save_ws = arcpy.env.workspace
        arcpy.env.workspace = out_ws
        
        #full path required or file won't be deleted correctly and rename fails
        out_path_fn = os.path.join(out_ws, out_fn)

        if self.debug: self.log_file.write ('Creating mosaic from blocks of %s files\n' % len(fns))
        if len(fns) > 1:
            arcpy.Mosaic_management(';'.join(fns[1:]), fns[0], "", "", "", nodata_value, "", "", "")
            #arc fails to remove these temporary files after Mosaic command so cleaning up
            t_fns = glob.glob(os.path.join(out_ws, 'x*.tif.vat.cpg'))           
            for t_fn in t_fns:
                if self.debug: self.log_file.write ('removing mosaic temp file t_fn=%s\n' % t_fn)
                os.remove(t_fn)
                
        if arcpy.Exists(out_path_fn):
            try:
                if self.debug: self.log_file.write ('%s exists...deleting\n' % out_path_fn)

                arcpy.Delete_management(out_path_fn)
            except:
                arcpy.AddError('Unable to delete the existing output file %s in %s. Make sure it is not in use.' % (out_fn, arcpy.env.workspace))

        if self.copy_landmask and src_tif:
            if self.debug: self.log_file.write ('Applying landmask to mosaic file %s and generating %s\n' % (fns[0], out_path_fn))

            data_ras = Raster(os.path.join(out_ws, fns[0]))
            
            if self.debug: self.log_file.write ('indata_dtype %s\n' % (self.indata_dtype))

            if np.issubdtype(self.indata_dtype, np.floating):
                out_raster = (Con(IsNull(Raster(src_tif)), data_ras, Con(Raster(src_tif) == self.landmask_value, self.landmask_value, data_ras)))
                out_pixel_type = "32_BIT_FLOAT"
            else:
                out_raster = Int(Con(IsNull(Raster(src_tif)), data_ras, Con(Raster(src_tif) == self.landmask_value, self.landmask_value, data_ras)))
                out_pixel_type = "8_BIT_UNSIGNED"

            tmp_fn = os.path.join(arcpy.env.scratchGDB, 'tmp_ras_%s' % os.getpid())
            out_raster.save(tmp_fn)
            
            #create tiff with pixel type based input data and set nodata
            arcpy.CopyRaster_management(tmp_fn, out_path_fn, "DEFAULTS", "", str(nodata_value), "", "", out_pixel_type)
            
            #delete temp file
            arcpy.Delete_management(tmp_fn)

            del out_raster

        else:
            if self.debug: self.log_file.write ('Renaming %s to %s\n' % (fns[0], out_path_fn))
            arcpy.Rename_management(fns[0], out_path_fn)

        #add color table for original file for integer type raster only
        desc_out = arcpy.Describe('%s\\Band_1' % out_path_fn)
        if desc_out.isInteger and src_tif:
            desc_src = arcpy.Describe('%s\\Band_1' % src_tif)
            if desc_src.isInteger:
                arcpy.AddColormap_management(out_path_fn, src_tif, "#")

        self.update_metadata(out_path_fn)
        
        arcpy.env.workspace = save_ws

    def update_metadata(self, fn):
        '''
        Add lineage metadata documenting the tool command 
        '''
        md_fn = fn + '.xml'
        
        if os.path.exists(md_fn):
            try:
                root = ET.parse(md_fn)
                e = root.find('Esri/DataProperties/lineage')
                if e is not None:
                    e.clear()
                    
                    dte = datetime.now()
                    args = ','.join(['%s="%s"' % (p.name, '' if p.value is None else p.valueAsText.lower() if isinstance(p.value, bool) else p.valueAsText) for p in self.parameters])
                        
                    attribs = {'Time': dte.strftime('%H%M%S'),
                              'Date': dte.strftime('%Y%m%d'),
                              'ToolSource': 'arcpy.%s (%s)' % (type(self).__name__, args)}

                    ET.SubElement(e, 'Process', attrib=attribs)
                    
                    root.write(md_fn)
                    
            except ImportError:
                self.report_error(logonly=True)
                arcpy.AddWarning('Unable to update metadata due to XML import error. Possible problem with python installation. Does not impact the composite results just missing some metadata.')
                pass

    def add_saps_metadata(self, comp_fn, fns):
        '''
        Extract relevant SAPS internal metadata from first source file and 
        propogate to composite file. 
        
        Note: only NOAA/NCCOS products contain internal SAPS metadata.
        '''        
        gdal.UseExceptions()
        
        ds = gdal.Open(fns[0], gdal.GA_ReadOnly)
        src_md = ds.GetMetadata()
        ds = None

        args = ','.join(['%s="%s"' % (p.name, '' if p.value is None else p.valueAsText.lower() if isinstance(p.value, bool) else p.valueAsText) for p in self.parameters])       
        cmd = 'arcpy.%s (%s)' % (type(self).__name__, args)
        
        ds = gdal.Open(comp_fn, gdal.GA_Update)                            
        comp_md = ds.GetMetadata()
        for k in src_md:
            if k.startswith('SAPS_product'):
                if k == 'SAPS_product_created':
                    comp_md[k] = f'{datetime.now():%Y%m%dT%H%M}'
                elif k == 'SAPS_product_desc':
                    comp_md[k] = f'RSTools (v{self.getversion()}) timeseries composite created by cmd: {cmd}.'
                elif k == 'SAPS_product_src':
                    comp_md[k] = ','.join(fns)
                    
                else:
                    comp_md[k] = src_md[k]
        ds.SetMetadata(comp_md)
        ds = None
        
    def output_stats(self, stats_data, block_no, out_fn, out_ws, out_dtype, x_cellsize, y_cellsize, map_llx, map_lly, nodata_value):
        '''
        '''
        save_ws = arcpy.env.workspace
        arcpy.env.workspace = out_ws

        if self.debug: self.log_file.write ('writing file %s\n\tblock_no=%s\n\txcell_size=%s\n\tlowerleft=(%s,%s)\n\tno_data=%s\n' % (out_fn, block_no, x_cellsize, map_llx, map_lly, nodata_value))
        if self.debug: self.log_file.write ('stats_data.shape=' + repr(stats_data.shape) + '\n')
        if self.debug: self.log_file.write ('type(stats_data)=%s  stats_data.dtype=%s\n' % (type(stats_data), stats_data.dtype))

        if ma.isMaskedArray(stats_data):
            if self.debug: self.log_file.write ('count stats_data masked values=%s stats_data.fill_value=%s\n' % (ma.count_masked(stats_data),  stats_data.fill_value))
            stats_data_filled = stats_data.filled(fill_value=nodata_value) if nodata_value else stats_data.filled()
            if self.debug: self.log_file.write ('stats_data_filled: amin=%s nanmin=%s amax=%s nanmax=%s\n' % (np.amin(stats_data_filled), np.nanmin(stats_data_filled), np.amax(stats_data_filled), np.nanmax(stats_data_filled)))

            if self.debug: self.log_file.write ('stats_data_filled.shape=' + repr(stats_data_filled.shape) + '\n')

        else:
            stats_data_filled = stats_data

        # Convert data block back to raster
        out_raster_block = arcpy.NumPyArrayToRaster(stats_data_filled.astype(out_dtype), arcpy.Point(map_llx, map_lly), x_cellsize, y_cellsize, nodata_value)

        # Save on disk temporarily as 't_filename_#.ext' or t_filename_# (if no extention)
        tmp_fn = 't_' + ('_%i.' % block_no).join(out_fn.rsplit('.', 1)) if '.' in out_fn else 't_%s_%i' % (out_fn, block_no)
        out_raster_block.save(os.path.join(out_ws, tmp_fn))

        del stats_data_filled
        del out_raster_block

        arcpy.env.workspace = save_ws

        return tmp_fn

    def get_time_range_end_date(self, start_dt):
        '''
        '''
        end_dt = start_dt + timedelta(self.duration-1)

        if self.do_not_span_months:
            #don't span month
            while (end_dt.month != start_dt.month):
                end_dt -= timedelta(1)
                if self.debug: self.log_file.write('Correcting month span: edt=%s\n' % end_dt)
        else:
            #we never span years
            while (end_dt.year != start_dt.year):
                end_dt -= timedelta(1)
                if self.debug: self.log_file.write('Correcting year span: edt=%s\n' % end_dt)

        return end_dt

    def get_end_date(self, start_dt):
        '''
        Return end date object for the period based on the start date object and comp_type.
        '''
        if self.comp_type == 'timerange':
            end_dt = self.get_time_range_end_date(start_dt)
            #check subsequent time period includes at least 80% of time window
            #   i.e. it isn't truncated by end of year, if so expand window to include residuals
            next_end_dt = self.get_time_range_end_date(end_dt  + timedelta(1))
            if self.debug: self.log_file.write('Testing residuals: edt=%s next_edt=%s**TEST=%s\n' % (end_dt, next_end_dt, (next_end_dt - end_dt) < timedelta(self.duration * .8)))
            if (next_end_dt - end_dt) < timedelta(self.duration * .8):
                end_dt = next_end_dt

        elif self.comp_type == 'monthly':
            end_day = calendar.monthrange(start_dt.year, start_dt.month)[1]
            end_dt = datetime(start_dt.year, start_dt.month, end_day)
        elif self.comp_type == 'yearly':
            end_dt = datetime(start_dt.year, 12, 31)
        elif self.comp_type == 'seasonal':
            end_dt = start_dt + relativedelta(months=+3) - timedelta(days=1)

        else:
            arcpy.AddError('Unsupported comp_type "%s".' % self.comp_type)

        return end_dt

class RasterStatsByPolygon(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""

            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            matchup_pnum = 8
            self.params[matchup_pnum+1].enabled = self.params[matchup_pnum].value
            # self.params[matchup_pnum+2].enabled = self.params[matchup_pnum].value
            # self.params[matchup_pnum+3].enabled = self.params[matchup_pnum].value
            #only supporting sample day matchups for now
            self.params[matchup_pnum+2].enabled = False
            self.params[matchup_pnum+3].enabled = False
            
            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))
                    
            matchup_pnum = 8
            if self.params[matchup_pnum].value:
                for i in [1, 2, 3]:
                    if not self.params[matchup_pnum+i].valueAsText:
                        self.params[matchup_pnum+i].setErrorMessage('ERROR: %s parameter is required if Perform date matchup parameter is checked' % (self.params[matchup_pnum+i].displayName))
            else:
                for i in [1, 2, 3]:
                    self.params[matchup_pnum+i].clearMessage()

            return

    def __init__(self):
        super(RasterStatsByPolygon, self ).__init__()
        self.label = 'Raster statistics by polygons'
        self.description = 'Extract statistics based on polygons within a feature class to a CSV file from one or more rasters.'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]

        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'

        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'

        # Zone_data
        param4 = arcpy.Parameter()
        param4.name = 'Zone_data'
        param4.displayName = 'Zone data'
        param4.parameterType = 'Required'
        param4.direction = 'Input'
        param4.datatype = 'Feature Layer'

        # Zone_field
        param5 = arcpy.Parameter()
        param5.name = 'Zone_field'
        param5.displayName = 'Zone field'
        param5.parameterType = 'Required'
        param5.direction = 'Input'
        param5.datatype = 'Field'
        param5.filter.list = ['Short', 'Long', 'Text']
        param5.parameterDependencies = [param4.name]

        # Statistic
        param6 = arcpy.Parameter()
        param6.name = 'Statistic'
        param6.displayName = 'Statistic'
        param6.parameterType = 'Required'
        param6.direction = 'Input'
        param6.datatype = 'String'
        param6.filter.list = ['ALL', 'MEAN', 'MAJORITY', 'MAXIMUM', 'MEDIAN', 'MINIMUM', 'MINORITY', 'RANGE', 'STD', 'SUM', 'VARIETY', 'MIN_MAX', 'MEAN_STD', 'MIN_MAX_MEAN']

##MU
        # Perform_date_matchup
        param8 = arcpy.Parameter()
        param8.name = 'Perform_date_matchup'
        param8.displayName = 'Perform date matchup'
        param8.parameterType = 'Required'
        param8.direction = 'Input'
        param8.datatype = 'Boolean'
        param8.value = 'false'

        # Date_attribute_name
        param9 = arcpy.Parameter()
        param9.name = 'Date_attribute_name'
        param9.displayName = 'Date attribute name'
        param9.parameterType = 'Optional'
        param9.direction = 'Input'
        param9.datatype = 'Field'
        param9.parameterDependencies = [param4.name]
        param9.filter.list = ['Date']

        # Allowed_time_difference
        param10 = arcpy.Parameter()
        param10.name = 'Allowed_time_difference'
        param10.displayName = 'Allowed time difference'
        param10.parameterType = 'Optional'
        param10.direction = 'Input'
        param10.datatype = 'Double'
        param10.value = '0'

        # Time_units
        param11 = arcpy.Parameter()
        param11.name = 'Time_units'
        param11.displayName = 'Time units'
        param11.parameterType = 'Optional'
        param11.direction = 'Input'
        param11.datatype = 'String'
        param11.value = 'day(s)'
        param11.filter.list = ['day(s)']
        # param11.filter.list = [u'day(s)', u'hour(s)']
##

        # Minimum valid_data value
        param12 = arcpy.Parameter()
        param12.name = 'Minimum_valid_data_value'
        param12.displayName = 'Minimum valid data value'
        param12.parameterType = 'Optional'
        param12.direction = 'Input'
        param12.datatype = 'Double'
        param12.value = '1'

        # Maximum valid_data value
        param13 = arcpy.Parameter()
        param13.name = 'Maximum_valid_data_value'
        param13.displayName = 'Maximum valid data value'
        param13.parameterType = 'Optional'
        param13.direction = 'Input'
        param13.datatype = 'Double'
        param13.value = '250'

        # Nodata_value
        param14 = arcpy.Parameter()
        param14.name = 'Nodata_value'
        param14.displayName = 'Nodata value'
        param14.parameterType = 'Required'
        param14.direction = 'Input'
        param14.datatype = 'Double'
        param14.value = '255'

        # Out_csv_file_name
        param15 = arcpy.Parameter()
        param15.name = 'Out_csv_file_name'
        param15.displayName = 'Out CSV file name'
        param15.parameterType = 'Required'
        param15.direction = 'Output'
        param15.datatype = 'File'
        param15.filter.list = ['csv']

        return [param0, param1, param2, param3, param1b, param4, param5, param6, param8, param9, param10, param11, param12, param13, param14, param15]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'raster_stats_by_poly'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)
            zstats_fns = []
            
            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.in_zone_data = parameters[5].valueAsText
            self.in_zone_field = parameters[6].valueAsText
            self.stat_name = parameters[7].valueAsText.upper()
           
##MU
            self.do_matchup = parameters[8].value
            self.match_date_fldname = parameters[9].valueAsText
            self.match_time = parameters[10].value
            self.match_units = parameters[11].valueAsText

            #parameter validation
            if self.do_matchup and not self.match_date_fldname: raise missingRequiredParameter('Date field name')
            if self.do_matchup and not self.match_time and self.match_time != 0: raise missingRequiredParameter('Allowed time difference')
            if self.do_matchup and not self.match_units: raise missingRequiredParameter('Time units')
##
            
            self.min_valid_val = parameters[12].value
            self.max_valid_val = parameters[13].value
            self.nodata_value = parameters[14].value
            full_out_csv_fn  = os.path.abspath(parameters[15].valueAsText)
            self.out_ws, self.out_csv_fn  = os.path.split(full_out_csv_fn )
            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_csv_fn  = self.validate_csv_name(self.out_csv_fn )

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                args = ','.join(['%s="%s"' % (p.name, '' if p.value is None else p.valueAsText.lower() if isinstance(p.value, bool) else p.valueAsText) for p in parameters])
                    
                self.log_file.write ('arcpy.%s (%s)\n' % (type(self).__name__, args))

                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)
           
            self.get_arc_lic('Spatial')

            in_files = self.get_files()
                
            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

##MU
            #only process files within matchup periods
            if in_files and self.do_matchup:
                in_files = self.match_up_file_filter(in_files, self.in_zone_data)
                if self.debug: self.log_file.write('after MU filter: number in_files=%s\n' % len(in_files))

            if not in_files:
                arcpy.AddWarning('No files found to process matching matchup dates...exiting.')
                return
##
            arcpy.env.overwriteOutput = True

            zstats_fns = self.zone_stats(in_files)
            self.zstat_to_csv(zstats_fns, in_files)
            
        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            for fn in zstats_fns:
                if arcpy.Exists(fn):
                    if self.debug: self.log_file.write ('deleting fn=%s\n' % fn)
                    arcpy.Delete_management(fn)

            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)
            arcpy.env.overwriteOutput = self.orig_overwriteoutput

    def zone_stats(self, fns):
        '''
        Generate zonal statistics for list of filenames

        Return list of zonal statistic filenames
        '''
        ignore_nodata = "DATA"

        zstats_fns = []
        self.out_fn_suffix = '_masked'

        work_dir = arcpy.env.scratchFolder if self.get_ws_type(self.in_ws) == 'FileSystem' else arcpy.env.scratchGDB
        
        in_zone_lyr = 'in_zone_lyr'
        arcpy.MakeFeatureLayer_management(self.in_zone_data, in_zone_lyr)          
        
        #assuming all rasters have same spatial properties as first in list
        arcpy.env.snapRaster = fns[0]
        arcpy.env.cellSize = fns[0]
        arcpy.env.outputCoordinateSystem = fns[0]
            
        for fn in fns:
            if self.debug: self.log_file.write ('creating zonal stats for %s.\n' % fn)

            fn_base = os.path.splitext(os.path.basename(fn))[0]
            out_fn = '{0}_zstats'.format(fn_base)

            #ArcGIS doesn't like '.' in filenames for GDB
            out_fn = out_fn.replace('.', self.ws_delimiter)
            out_tn = arcpy.ValidateTableName(out_fn, arcpy.env.scratchGDB)
            if self.debug: self.log_file.write ('outputing zonal stats to %s.\n' % os.path.join(arcpy.env.scratchGDB, out_tn))

            #create raster of only valid data
            masked_fn_list = self.create_masks([fn], mask_dir=work_dir)
            masked_fn = masked_fn_list[0] if masked_fn_list else fn
            
            try:
                arcpy.AddMessage('Extracting statistics for %s' % fn)
                if self.debug: self.log_file.write ('env.workspace=%s\n' % arcpy.env.workspace)
##MU                
                if self.do_matchup:
                    (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(fn)
                    fn_date = datetime.strptime('%s%s%s %s' % (year, mm, dd, hhmm), '%Y%m%d %H%M')
#                    fn_date = '%Y-%m-%d %H%M:00'.format(year, mm, dd, hhmm[0:2], hhmm[2:4])
                    field = arcpy.AddFieldDelimiters(self.in_zone_data, self.match_date_fldname)
##TODO hours days                    
#                    query = "{0} = date '{1:%Y-%m-%d %H%M:00}'".format(field, fn_date)
                    query = "{0} = date '{1:%Y-%m-%d 00:00:00}'".format(field, fn_date)
                    if self.debug: self.log_file.write ('query=%s\n' % query)
                    arcpy.SelectLayerByAttribute_management(in_zone_lyr, 'NEW_SELECTION', query)
                    
                    out_zonal_stats = ZonalStatisticsAsTable(in_zone_lyr, self.in_zone_field, masked_fn, out_tn, ignore_nodata, self.stat_name)                   
##
                else:  
                    out_zonal_stats = ZonalStatisticsAsTable(in_zone_lyr, self.in_zone_field, masked_fn, out_tn, ignore_nodata, self.stat_name)                   

                zstats_fns.append(out_tn)
                if masked_fn != fn:
                    arcpy.Delete_management(masked_fn)
                    
                if self.do_matchup:
                    arcpy.SelectLayerByAttribute_management(in_zone_lyr, 'CLEAR_SELECTION')
                    
                del out_zonal_stats                

            except arcpy.ExecuteError:
                msgs = arcpy.GetMessages(2)

                arcpy.AddError(msgs)
                if self.log_file: self.log_file.write(msgs)

                arcpy.AddError("Unable to extract statistics from %s. Verify that the raster overlaps the (selected) features in the zone data %s." % (fn, self.in_zone_data))

            except:
                self.report_error()

        arcpy.env.snapRaster = ''
        arcpy.env.cellSize = None
        arcpy.env.outputCoordinateSystem = None
                
        in_zone_lyr = None
        
        return zstats_fns

    def raster_cell_size (self, raster_fn):
        '''
        Return mean cell width for first band of raster_fn
        '''
        desc = arcpy.Describe('%s\\Band_1' % raster_fn)
        if self.debug: self.log_file.write ('raster cell size=%s.\n' % desc.meanCellWidth)
        return desc.meanCellWidth


    def zone_geom(self, cell_size, geom_type='CENTROID'):
        '''
        Calculate zonal geometry as a table.

        Return table name
        '''
        fn_base = os.path.splitext(os.path.basename(self.in_zone_data))[0]
        zone_geom_fn = os.path.join(arcpy.env.scratchGDB, '%s_zg' % fn_base)
        if self.debug: self.log_file.write ('creating zonal geometry in %s.\n' % zone_geom_fn)
        if self.debug: self.log_file.write ('zonal geometry params %s, %s, %s, %s.\n' % (self.in_zone_data, self.in_zone_field, zone_geom_fn, cell_size) )
        ZonalGeometryAsTable(self.in_zone_data, self.in_zone_field, zone_geom_fn, cell_size)

        return zone_geom_fn

    def get_zone_centroid(self, zone_geom_fn):
        '''
        Return dictionary keyed with zoneid. Dictionary items are list of centroid x, centroid y
        '''
        zone_geom_dict = {}

        with arcpy.da.SearchCursor(zone_geom_fn, 'VALUE;XCENTROID;YCENTROID') as cursor:
            for row in cursor:
                zone_geom_dict[row.getValue("VALUE")] = [row.getValue("XCENTROID"), row.getValue("YCENTROID")]

        return zone_geom_dict

    def get_zstats_in_fn(self, zstats_fn, in_files):
        '''
        Return the in filename from which the zstat_fn was built.
        '''

        for fn in in_files:
            fn_base = os.path.splitext(os.path.basename(fn))[0]
            test_fn_base = '{0}_zstats'.format(fn_base).replace('.', self.ws_delimiter)
            test_tn = arcpy.ValidateTableName(test_fn_base, arcpy.env.scratchGDB)

            if zstats_fn == test_tn:
                return fn

    def zstat_to_csv(self, zstats_fns, in_files):
        '''
        Combine list zonal statistic output files into a single CSV file.
        '''
        if self.debug: self.log_file.write ('Combining zonal stats into CSV file %s\n' % self.out_csv_fn)
        csv_file = codecs.open(os.path.join(self.out_ws, self.out_csv_fn), "w", encoding="utf-8")

        do_hdr = True
        for fn in zstats_fns:
            in_fn = self.get_zstats_in_fn(fn, in_files)
            (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(in_fn)

            row_cnt = 0
            with arcpy.da.SearchCursor(fn, '*') as cursor:
                if do_hdr:
                    #build header
                    if self.debug: self.log_file.write ('type(zonal stats fields)=%s\n' % type(cursor.fields))
                    if self.debug: self.log_file.write ('zonal stats fields={0}\n'.format(','.join(cursor.fields)))

                    hdr = 'date'

                    #add hdr columns for scaled and unscaled values
                    start_data_val_idx = cursor.fields.index('AREA')+1
                    if self.debug: self.log_file.write ('start_data_val_idx=%s\n' % start_data_val_idx)

                    #assuming first file matches others
                    can_unscale = self.can_unscale(in_files[0])

                    #if projected coord system get info so can convert "SUM"
                    # to SUM_area_sqkm
                    desc = arcpy.Describe('%s\\Band_1' % in_files[0])
                    sr = desc.SpatialReference
                    if sr.PCSName:
                        meters_per_init = sr.metersPerUnit
                        x_cell_size = desc.meanCellWidth
                        y_cell_size = desc.meanCellHeight
                        area2sqm = x_cell_size * y_cell_size * meters_per_init * meters_per_init
                        area2sqkm = area2sqm / (1000 * 1000)
                        if self.debug: self.log_file.write (f'area2sqkm={area2sqkm}\n')
                    else:
                        area2sqkm = None
        
                    num_stat_cols = 0
                    for i, fld in enumerate(cursor.fields):
                        hdr += ',' + fld
                        num_stat_cols += 1
                        if ((i >= start_data_val_idx) and
                            (fld in ['MAX', 'MEDIAN', 'MIN'])):
                            if can_unscale:
                                hdr += ',unscaled_{0}'.format(fld)
                                num_stat_cols += 1
                        if ((i >= start_data_val_idx) and
                            (fld in ['SUM'])):
                            if area2sqkm is not None:
                                hdr += ',SUM_area_sqkm'
                                num_stat_cols += 1
                                
                    hdr += ',file'

                    #use the first filename to identify type of files (assuming other are the same)
                    if prod_ver is not None:
                        hdr += ',prod_ver'
                        saps_prodfile = True
                    else:
                        saps_prodfile = False
                        
                    csv_file.write(hdr + '\n')

                    do_hdr = False
                    
                for row in cursor:
                    out_line = '{0}-{1}-{2},'.format(mm, dd, year)
                    for i, fld in enumerate(row):
                        if ',' in str(fld):
                            out_line += '"{0}",'.format(fld)
                        else:
                            out_line += '{0},'.format(fld)
                        if ((i >= start_data_val_idx) and
                            (cursor.fields[i] in ['MAX', 'MEDIAN', 'MIN'])):

                            if self.debug: self.log_file.write ('fld={0} prod_name={1}  prod_ver={2}\n'.format(fld, prod_name, prod_ver))
                            if can_unscale:
                                unscaled_val = self.get_unscaled_val(fld, prod_name, prod_ver=prod_ver)
                                if unscaled_val:
                                    out_line += '{0:.5f},'.format(unscaled_val)
                                else:
                                    out_line += ','

                        if ((i >= start_data_val_idx) and
                            (cursor.fields[i] in ['SUM'])):
                            if area2sqkm is not None:
                                sum_area_sqkm = fld * area2sqkm
                                out_line += '{0:.5f},'.format(sum_area_sqkm)

                    out_line += '{0}'.format(in_fn)

                    if saps_prodfile:
                        #include SAPS product version
                        out_line += ',{0}'.format(prod_ver)
                        
                    csv_file.write(out_line + '\n')
                    row_cnt += 1

                if row_cnt == 0:
                    init_stats = ','.join(['0'] * num_stat_cols)
                    out_line = f'{mm}-{dd}-{year},{init_stats},{in_fn}'
                    csv_file.write(out_line + '\n')
                    arcpy.AddMessage(f'{in_fn} has no valid data pixels within zones. Setting all results to 0.')
                    if self.debug: self.log_file.write(f'Info msg: {in_fn} has no valid data pixels within zones. Setting all results to 0.\n')
        csv_file.close()
        del csv_file

class MaskNoData(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""

            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            suffix_pnum = 7
            if self.params[suffix_pnum].value:
                if self.params[suffix_pnum].value[0] != '_':
                    self.params[suffix_pnum].value = '_' + self.params[suffix_pnum].value
            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            return

    def __init__(self):
        super(MaskNoData, self ).__init__()

        self.label = 'Mask no data values'
        self.description = 'Mask no data values of raster files by setting to Null'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]

        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'

        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'

        # Minimum valid_data value
        param4 = arcpy.Parameter()
        param4.name = 'Minimum_valid_data_value'
        param4.displayName = 'Minimum valid data value'
        param4.parameterType = 'Optional'
        param4.direction = 'Input'
        param4.datatype = 'Double'
        param4.value = '1'

        # Maximum valid_data value
        param5 = arcpy.Parameter()
        param5.name = 'Maximum_valid_data_value'
        param5.displayName = 'Maximum valid data value'
        param5.parameterType = 'Optional'
        param5.direction = 'Input'
        param5.datatype = 'Double'
        param5.value = '250'

        # Output_filename_suffix
        param6 = arcpy.Parameter()
        param6.name = 'Output_filename_suffix'
        param6.displayName = 'Output filename suffix'
        param6.parameterType = 'Optional'
        param6.direction = 'Input'
        param6.datatype = 'String'
        param6.value = '_masked'

        # Output_workspace
        param7 = arcpy.Parameter()
        param7.name = 'Output_workspace'
        param7.displayName = 'Output workspace'
        param7.parameterType = 'Required'
        param7.direction = 'Input'
        param7.datatype = ('Workspace')

        # Nodata_value
        param8 = arcpy.Parameter()
        param8.name = 'Nodata_value'
        param8.displayName = 'Nodata value'
        param8.parameterType = 'Required'
        param8.direction = 'Input'
        param8.datatype = 'Double'
        param8.value = '255'

        return [param0, param1, param2, param3, param1b, param4, param5, param6, param7, param8]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'mask_nodata'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.min_valid_val = parameters[5].value
            self.max_valid_val = parameters[6].value
            self.out_fn_suffix = self.validate_fn_suffix(parameters[7].valueAsText)
            self.out_ws = parameters[8].valueAsText
            self.nodata_value = parameters[9].value

            if self.min_valid_val is None and self.max_valid_val is None:
                arcpy.AddError('Please specify the minimum and/or maximum valid data values...exiting.')
                return

            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_ext = self.get_out_ext(self.out_ws_type)

            self.get_arc_lic('Spatial')

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\nin_ws=%s\nin_filter=%s\nrecursive=%s\nout_fn_suffix=%s\nout_ws=%s\nout_ext=%s\nmin_valid_val=%s\nmax_valid_val=%s\nnodata_val=%s\n' % (self.in_ws, self.in_filter, self.recursive, self.out_fn_suffix, self.out_ws, self.out_ext, self.min_valid_val, self.max_valid_val, self.nodata_value))
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)

            # arcpy.env.workspace = self.in_ws

            in_files = self.get_files()

            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            valid = self.validate_nodata(self.nodata_value, in_files[0])
            if not valid:
                arcpy.AddError('Invalid nodata value...exiting.')
                return

            mask_fns = self.create_masks(in_files, mask_dir=self.out_ws)

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)

            arcpy.env.overwriteOutput = self.orig_overwriteoutput

class PixelExtractByPoint(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""

            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            matchup_pnum = 10
            self.params[matchup_pnum+1].enabled = self.params[matchup_pnum].value
            self.params[matchup_pnum+2].enabled = self.params[matchup_pnum].value
            self.params[matchup_pnum+3].enabled = self.params[matchup_pnum].value
            
            extract_window_pnum = 14
            #disable this option if multi-pixel point extract
            if self.params[extract_window_pnum].value in ['3x3', '5x5', '7x7', '9x9', '11x11', '13x13']:
                self.params[extract_window_pnum+3].enabled = False
            else:
                self.params[extract_window_pnum+3].enabled = True
                
            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            band_filt_pnum = 5
            if self.params[band_filt_pnum].value:
                pat = r'^(([0-9]+,)|([0-9]+-[0-9]+,))*(([0-9]+)|([0-9]+-[0-9]+))$'
                if re.match(pat, self.params[band_filt_pnum].value):
#                if re.match(r'^[0-9-,]*$', self.params[band_filt_pnum].value):
                    self.params[band_filt_pnum].clearMessage()
                else:                
                    self.params[band_filt_pnum].setErrorMessage('Only numeric, "," and "-" values valid.')
                    
            matchup_pnum = 10
            if self.params[matchup_pnum].value:
                for i in [1, 2, 3]:
                    if not self.params[matchup_pnum+i].valueAsText:
                        self.params[matchup_pnum+i].setErrorMessage('ERROR: %s parameter is required if Perform date matchup parameter is checked' % (self.params[matchup_pnum+i].displayName))
            else:
                for i in [1, 2, 3]:
                    self.params[matchup_pnum+i].clearMessage()
                                        
            return

    def __init__(self):
        super(PixelExtractByPoint, self).__init__()
        self.label = 'Pixel extract by points'
        self.description = 'Extract pixels values from one or more rasters to a CSV file based on point feature class.'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]

        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'

        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'

        # Band filter
        param_band_filt = arcpy.Parameter()
        param_band_filt.name = 'Band_filter'
        param_band_filt.displayName = 'Band_filter'
        param_band_filt.parameterType = 'Optional'
        param_band_filt.direction = 'Input'
        param_band_filt.datatype = 'String'
#        param_band_filt.value = u'1'

        # Input_point_feature_class
        param4 = arcpy.Parameter()
        param4.name = 'Input_point_feature_class'
        param4.displayName = 'Input point feature class'
        param4.parameterType = 'Required'
        param4.direction = 'Input'
        param4.datatype = 'Feature Layer'
        param4.filter.list = ["Point"]

        # Interpolate_values
        param5 = arcpy.Parameter()
        param5.name = 'Interpolate_values'
        param5.displayName = 'Interpolate values'
        param5.parameterType = 'Required'
        param5.direction = 'Input'
        param5.datatype = 'Boolean'
        param5.value = 'false'
        ##since we re-wrote the pixel extraction routine we're currently
        #   not supporting this option. This will hide it from user.
        param5.parameterType = 'Derived'

        # Point_field
        param6 = arcpy.Parameter()
        param6.name = 'Point_field'
        param6.displayName = 'Point attribute(s)'
        param6.parameterType = 'Optional'
        param6.direction = 'Input'
        param6.datatype = 'Field'
        param6.multiValue = True
        param6.filter.list = ['OID', 'Short', 'Long', 'Single', 'Double', 'Text', 'Date']
        param6.parameterDependencies = [param4.name]

        # Out_csv_file_name
        param7 = arcpy.Parameter()
        param7.name = 'Out_csv_file_name'
        param7.displayName = 'Out CSV file name'
        param7.parameterType = 'Required'
        param7.direction = 'Output'
        param7.datatype = 'File'
        param7.filter.list = ['csv']

        # Perform_date_matchup
        param8 = arcpy.Parameter()
        param8.name = 'Perform_date_matchup'
        param8.displayName = 'Perform date matchup'
        param8.parameterType = 'Required'
        param8.direction = 'Input'
        param8.datatype = 'Boolean'
        param8.value = 'false'

        # Date_attribute_name
        param9 = arcpy.Parameter()
        param9.name = 'Date_attribute_name'
        param9.displayName = 'Date attribute name'
        param9.parameterType = 'Optional'
        param9.direction = 'Input'
        param9.datatype = 'Field'
        param9.parameterDependencies = [param4.name]
        param9.filter.list = ['Date']

        # Allowed_time_difference
        param11 = arcpy.Parameter()
        param11.name = 'Allowed_time_difference'
        param11.displayName = 'Allowed time difference'
        param11.parameterType = 'Optional'
        param11.direction = 'Input'
        param11.datatype = 'Double'
        param11.value = '1'

        # Time_units
        param12 = arcpy.Parameter()
        param12.name = 'Time_units'
        param12.displayName = 'Time units'
        param12.parameterType = 'Optional'
        param12.direction = 'Input'
        param12.datatype = 'String'
        param12.value = 'day(s)'
        param12.filter.list = ['day(s)', 'hour(s)']

        #
        param_ext_window = arcpy.Parameter()
        param_ext_window.name = 'Extract_pixel_window'
        param_ext_window.displayName = 'Extract pixel window'
        param_ext_window.parameterType = 'Required'
        param_ext_window.direction = 'Input'
        param_ext_window.datatype = 'String'
        param_ext_window.filter.list = ['1x1', '3x3', '3x3 min', '3x3 max', '3x3 mean', '5x5', '5x5 min', '5x5 max', '5x5 mean', '7x7', '9x9', '11x11', '13x13']
        param_ext_window.value = param_ext_window.filter.list[0]

        # Minimum valid_data value
        param13 = arcpy.Parameter()
        param13.name = 'Minimum_valid_data_value'
        param13.displayName = 'Minimum valid data value'
        param13.parameterType = 'Optional'
        param13.direction = 'Input'
        param13.datatype = 'Double'
        param13.value = '1'

        # Maximum valid_data value
        param14 = arcpy.Parameter()
        param14.name = 'Maximum_valid_data_value'
        param14.displayName = 'Maximum valid data value'
        param14.parameterType = 'Optional'
        param14.direction = 'Input'
        param14.datatype = 'Double'
        param14.value = '250'
              
        # Multi-bands to columns value
        param15 = arcpy.Parameter()
        param15.name = 'bands_to_cols'
        param15.displayName = 'Multi-band as columns '
        param15.parameterType = 'Optional'
        param15.direction = 'Input'
        param15.datatype = 'Boolean'
        param15.value = 'False'

        return [param0, param1, param2, param3, param1b, param_band_filt, param4, param5, param6, param7, param8, param9, param11, param12, param_ext_window, param13, param14, param15]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'pixel_extract_by_point'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.band_filter = parameters[5].valueAsText

            self.orig_in_point_fc = parameters[6].valueAsText
            self.interp_values = parameters[7].value
            self.pt_attribnames = parameters[8].valueAsText.split(';') if parameters[8].value else []
            full_out_csv_fn  = os.path.abspath(parameters[9].valueAsText)
            self.out_ws, self.out_csv_fn  = os.path.split(full_out_csv_fn )
            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_csv_fn  = self.validate_csv_name(self.out_csv_fn )

            self.do_matchup = parameters[10].value
            self.match_date_fldname = parameters[11].valueAsText
            self.match_time = parameters[12].value
            self.match_units = parameters[13].valueAsText

            #parameter validation
            if self.do_matchup and not self.match_date_fldname: raise missingRequiredParameter('Date field name')
            if self.do_matchup and not self.match_time and self.match_time != 0: raise missingRequiredParameter('Allowed time difference')
            if self.do_matchup and not self.match_units: raise missingRequiredParameter('Time units')

            self.ext_window = parameters[14].valueAsText

            self.min_valid_val = parameters[15].value
            self.max_valid_val = parameters[16].value
            
            #override setting if not supported for extract window option
            self.bands_to_cols = parameters[17].value if self.ext_window not in ['3x3', '5x5', '7x7', '9x9', '11x11', '13x13'] else False

            if (self.interp_values and
                (self.min_valid_val is not None or self.max_valid_val is not None)):
                arcpy.AddWarning('You checked the "Interpolate values" option which requires the input files must have all invalid data pixels defined as "nodata". If this is not the case, you can use the "Mask nodata" tool for setting invalid data values to "nodata".')

            # self.get_arc_lic('Spatial')

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\nselection method=%s\nin_ws=%s\nin_filter=%s\nrecursive=%s\nin_flist=%s\nin_point_fc=%s\ninterp=%s\nout_point_fc=%s\nout_ws=%s\n' % (self.input_method, self.in_ws, self.in_filter, self.recursive, self.in_filelist_fn, self.orig_in_point_fc, self.interp_values, self.out_csv_fn, self.out_ws))
                self.log_file.write('band_filter={0}\n'.format(self.band_filter))
                self.log_file.write('pt_attribnames={0}\n'.format(self.pt_attribnames))
                self.log_file.write('do_matchup=%s\n' % self.do_matchup)
                self.log_file.write('match_date_fldname={0}\n'.format(self.match_date_fldname))
                self.log_file.write('match_time=%s\n' % self.match_time)
                self.log_file.write('match_units=%s\n' % self.match_units)
                self.log_file.write('ext_window=%s\n' % self.ext_window)
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('min_valid_val=%s\n' % (self.min_valid_val))
                self.log_file.write ('max_valid_val=%s\n' % (self.max_valid_val))
                self.log_file.write ('bands_to_cols=%s\n' % (self.bands_to_cols))
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)

            # arcpy.env.workspace = self.in_ws

            in_files = self.get_files()

            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            #only process files within matchup periods
            if in_files and self.do_matchup:
                in_files = self.match_up_file_filter(in_files, self.orig_in_point_fc)
                if self.debug: self.log_file.write('after MU filter: number in_files=%s\n' % len(in_files))

            if not in_files:
                arcpy.AddWarning('No files found to process matching matchup dates...exiting.')
                return

            self.extract(in_files)
            del in_files

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except invalidParameter as e:
            arcpy.AddError('Invalid parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.debug: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.debug: self.log_file.close()
            # for extension_name in self.lic_checked_out:
                # if self.lic_checked_out[extension_name]:
                    # self.return_arc_lic(extension_name)

            arcpy.env.overwriteOutput = self.orig_overwriteoutput

    def clean_up(self):
        '''
        '''
        self.delete_copy_in_point_fc()

    def delete_copy_in_point_fc(self):
        '''
        '''
        orig_in_point_fc_base = os.path.split(self.orig_in_point_fc)[1]

        in_point_fc_full_fn = os.path.join(arcpy.env.scratchGDB, os.path.splitext(orig_in_point_fc_base)[0])
        if arcpy.Exists(in_point_fc_full_fn):
            arcpy.Delete_management(in_point_fc_full_fn)

    def build_in_raster_vt(self, in_files):
        '''
        Build value table based on format required by ExtractMultiValuesToPoints. (i.e filename and fieldname)
        '''
        vt = arcpy.ValueTable(2)
        for i, fn in enumerate(in_files):
            vt.addRow('"%s" fn_%s' % (fn, i))

        if self.debug: self.log_file.write ('in_rasters_vt=%s\n' % vt.exportToString())

        self.in_rasters_vt = vt
       
    def get_bandlist(self, band_filter, num_bands):
        '''
        Converts band filter string to a list of band indices (zero_based). If 
        no band_filter return list containing all band indices.
        
        Expected band filter string format:  
            comma-separated list of one or more band numbers (one-based) or one or more
            band ranges (defined by <start_band_num>-<end_ band_num>)
           
        '''
        try:
            if band_filter:
                band_dict = {}
                items = band_filter.split(',')
                
                for item in items:
                    if '-' in item:
                        b1, b2 = item.split('-')
                        for bnum in range(int(b1)-1, int(b2)):
                            if bnum < num_bands:
                                band_dict[bnum] = True
                            else:
                                arcpy.AddWarning('Band filter (%s) greater than num_bands (%s) in input raster. Ignored' % (bnum+1, num_bands))
                                if self.debug: self.log_file.write ('Band filter (%s) greater than num_bands (%s) in input raster. Ignored.\n' % (bnum+1, num_bands))
                    else:
                        bnum = int(item)-1
                        if bnum < num_bands:
                            band_dict[bnum] = True
                        else:
                            arcpy.AddWarning('Band filter (%s) greater than num_bands (%s) in input raster. Ignored' % (bnum+1, num_bands))
                            if self.debug: self.log_file.write ('Band filter (%s) greater than num_bands (%s) in input raster. Ignored.\n' % (bnum+1, num_bands))
                return sorted(band_dict.keys())
            else:
                #no filtering - all bands
                return list(range(num_bands))
                
        except Exception:
            # Get the traceback object
            extype, exval, extr = sys.exc_info()
            exception_list = traceback.format_exception(extype, exval, extr)

            # Concatenate information together concerning the error into a message string
            pymsg = "PYTHON ERRORS:\nTraceback info:\n" + '\n'.join(exception_list)
            msgs = "ArcPy ERRORS:\n" + arcpy.GetMessages(2) + "\n"

            if self.log_file:
                self.log_file.write(pymsg + "\n")
                self.log_file.write(msgs)

            raise invalidParameter('Band filter format not recognized. Provide a comma-separated list of band numbers and/or band ranges (start & end band number separated by "-".')
        
    def extract(self, in_files):
        '''
        Extract raster values for all raster in in_files list. Raster values to
        extract are based on a point feature class. Output is written to
        csv file.

        Initially used ExtractMultiValuesToPoints but re-wrote after identified  memory
        leak issues produced invalid results.
        '''
        num_files = len(in_files)
        arcpy.AddMessage('Extracting pixel values from %s files' % num_files)

        ##reformat output to standard drill file format
        hdr = 'date,x,y'

        window_size = int(self.ext_window.split('x')[0])
        window_stat = None if len(self.ext_window.split()) == 1 else self.ext_window.split()[1]

        #assuming first file matches others
        can_unscale = self.can_unscale(in_files[0])
            
        if (window_size > 1 and not window_stat):
            for i in range(1, window_size+1):
                for j in range(1, window_size+1):
                    hdr += ',%sx%s_val' % (i, j)
                    if can_unscale:
                        hdr += ',%sx%s_unscaled_val' % (i, j)
        else:
            hdr += ',val'
            if can_unscale:
                hdr += ',unscaled_val'
        
        hdr += ',band_num,file'
                
        orig_in_point_fc_base = os.path.split(self.orig_in_point_fc)[1]

        desc = arcpy.Describe(self.orig_in_point_fc)
        flds = ["SHAPE@"]
        fld_idx_shape = 0
        in_point_fc_sr = desc.spatialReference

        offset = 1

        if self.do_matchup:
            flds.append(self.match_date_fldname)
            hdr += ',date_delta_({0})'.format(self.match_units)
            fld_idx_match_date = offset

            offset += 1

        if self.pt_attribnames:
            for pt_attribname in self.pt_attribnames:
                flds.append(pt_attribname)
                hdr += ',{0}'.format(pt_attribname)
            fld_idx_pt_attrib = offset
            offset += 1

        arcpy.env.overwriteOutput = True
        gcs_wgs_sr = arcpy.SpatialReference(4326)
        
        out_csv_path_fn = os.path.join(self.out_ws, self.out_csv_fn)
        csv_file = codecs.open(out_csv_path_fn, "w", encoding="utf-8")

        for cnt, fn in enumerate(in_files):
            if self.debug: self.log_file.write ('processing (%s of %s) raster from %s\n' % (cnt+1, num_files, fn))
            arcpy.AddMessage('Processing (%s of %s) raster from %s' % (cnt+1, num_files, fn))
            
            (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(fn)
            
            if hdr is not None:
                #use the first filename to identify type of files (assuming other are the same)
                if prod_ver is not None:
                    hdr += ',prod_ver'
                    saps_prodfile = True
                else:
                    saps_prodfile = False
                    
                csv_file.write(hdr + '\n')
                hdr = None
            
            in_raster = arcpy.Raster(fn)
            in_raster_sr = in_raster.spatialReference

            nodata_val = in_raster.noDataValue if in_raster.noDataValue else None
            if self.debug: self.log_file.write ('nodata_val=%s\n' % (nodata_val))

            #raster extent in map units
            map_llx = in_raster.extent.XMin
            map_lly = in_raster.extent.YMin
            map_urx = in_raster.extent.XMin + in_raster.width * in_raster.meanCellWidth
            map_ury = in_raster.extent.YMin + in_raster.height * in_raster.meanCellHeight

            if self.debug: self.log_file.write ('Raster extent: llx=%s lly=%s urx=%s ury=%s\n' % (map_llx, map_lly, map_urx, map_ury))

            num_bands = in_raster.bandCount
            band_list = self.get_bandlist(self.band_filter, num_bands)
                
            fn_date = datetime.strptime( '%s%s%s %s' % (year, mm, dd, hhmm), '%Y%m%d %H%M')

            with arcpy.da.SearchCursor(self.orig_in_point_fc, flds) as cursor:
                for row in cursor:

                    if self.do_matchup:
                        pt_date = row[fld_idx_match_date]
                        #skip point if not valid date or fn date not within match period
                        if ((pt_date is None) or 
                            (not self.test_matchup(fn_date, pt_date))):
                            continue
                        if self.debug: self.log_file.write ('pt_date=%s \n' % (pt_date))

                    in_feat = row[fld_idx_shape]
                    if in_feat is None:
                        arcpy.AddWarning('Bad geometry for point skipping. Point info (%s): %s' % (str(flds), str(row)))
                        continue
                    in_pnt = in_feat.getPart()

                    #reproject point to raster map units
                    feat = in_feat.projectAs(in_raster_sr)
                    pnt = feat.getPart()
                    if self.debug: self.log_file.write ('in pnt: x=%s y=%s pnt:x=%s y=%s\n' % (in_pnt.X, in_pnt.Y, pnt.X, pnt.Y))

                    #skip point if outside raster extent
                    if (not (map_llx <= pnt.X <= map_urx) or
                        not (map_lly <= pnt.Y <= map_ury)):
                       continue

                    # Make points in window
                    ext_pnts = self.get_ext_points(pnt, in_raster.meanCellWidth, window_size)

                    if self.debug: self.log_file.write ('ext_pnts=%s\n' % ext_pnts)
    
                    #get all pixels in window
                    #  RasterToNumPyArray expects lower left cell corner (it'll auto snap to nearest cell corner)
                    llc_pnt = ext_pnts['%s,%s' % (window_size-1, 0)]
                    llc_pnt.X -= in_raster.meanCellWidth/2.
                    llc_pnt.Y -= in_raster.meanCellWidth/2.
                    
                    pnt_data_in_window = arcpy.RasterToNumPyArray(in_raster, llc_pnt, window_size, window_size).reshape(num_bands, window_size, window_size)
                    
                    #summarize pixels in window based on statistic                   
                    if window_stat is not None:
                        #"RuntimeWarning: <stat> of empty slice" was being reporting tool failed ERROR so suppressing
                        with warnings.catch_warnings(): 
                            warnings.simplefilter("ignore") 
                            # if self.debug: self.log_file.write ('pnt_data_in_window=%s\n' %  (pnt_data_in_window))
                            pnt_data_in_window = pnt_data_in_window.astype(np.float32)
                            
                            #set pixel values outside valid range to nan
                            if self.min_valid_val is not None:
                                pnt_data_in_window[pnt_data_in_window < self.min_valid_val] = np.nan
                            if self.max_valid_val is not None:
                                pnt_data_in_window[pnt_data_in_window > self.max_valid_val] = np.nan
                            
                            #need to handle multiband files
                            stat_axes = None if num_bands == 1 else (1, 2)
                            
                            if window_stat == 'min':
                                pnt_data_in_window = np.nanmin(pnt_data_in_window, axis=stat_axes).reshape((num_bands, 1, 1))                            
                            elif window_stat == 'max':
                                pnt_data_in_window = np.nanmax(pnt_data_in_window, axis=stat_axes).reshape((num_bands, 1, 1))

                            elif window_stat == 'mean':
                                pnt_data_in_window = np.nanmean(pnt_data_in_window, axis=stat_axes).reshape((num_bands, 1, 1))
                                    
                    # if self.debug: self.log_file.write ('pnt_data_in_wind=%s\n' % pnt_data_in_window)
                    if self.debug: self.log_file.write ('pnt_data_in_wind.shape=' + repr(pnt_data_in_window.shape) + ' num_bands=%s' % (num_bands) + ' window_size=%s\n' % (window_size))

                    for band_idx in band_list:
                        
                        # reproject point to lat/lon for output coords based on center of extract window
                        pnt_geom = arcpy.PointGeometry(ext_pnts['%s,%s' % (int(window_size/2), int(window_size/2))], in_raster_sr)
                        out_feat = pnt_geom.projectAs(gcs_wgs_sr)
                        out_pnt = out_feat.getPart()
                        out_line = '{0}-{1}-{2},{3},{4}'.format(mm, dd, year, out_pnt.X, out_pnt.Y, )

                        for data_wind_row, cols in enumerate(pnt_data_in_window[band_idx]):
                            if self.debug: self.log_file.write ('cols.shape=' + repr(cols.shape) + 'type(cols)=%s' % (type(cols)) + ' data_wind_row=%s\n' % (data_wind_row))

                            for data_wind_col, pnt_data in enumerate(cols):

                                # if self.debug: self.log_file.write ('pnt_data=%s\n' % pnt_data)
                                if self.debug: self.log_file.write ('pnt_data.shape=' + repr(pnt_data.shape) + 'type(pnt_data)=%s' % (type(pnt_data)) + ' num_bands=%s\n' % (num_bands))

                                val = '' if np.isnan(pnt_data) else self.validate_data(pnt_data, nodata_val=nodata_val)
                                
                                out_line += ',{0}'.format(val)
                                if can_unscale:
                                    unscaled_val = self.get_unscaled_val(val, prod_name, prod_ver=prod_ver)                                   
                                    out_line += ',{0}'.format(unscaled_val)
                           
                        out_line += ',{0},{1}'.format(band_idx+1, fn)

                        if self.do_matchup:
                            #add date from point shapefile
                            if self.debug: self.log_file.write ('%s**%s\n' % (fn_date, pt_date))
                            if 'day' in self.match_units:
                                #report days difference (ignore file time)
                                td = pt_date.date() - fn_date.date()
                                out_line += ',{0}'.format(td.days)
                            else:
                                #report hours difference
                                td = pt_date - fn_date
                                out_line += ',{0}'.format(td.seconds/60./60.)

                        for i, pt_attribname in enumerate(self.pt_attribnames): 
                            fc_fields = arcpy.ListFields(self.orig_in_point_fc, pt_attribname)
                            #quote string fields to avoid potential problems with commas
                            if fc_fields[0].type == 'String':
                                out_line += ',"{0}"'.format(row[fld_idx_pt_attrib+i])
                            else:
                                out_line += ',{0}'.format(row[fld_idx_pt_attrib+i])
                            

                        if saps_prodfile:
                            #include SAPS product version
                            out_line += ',{0}'.format(prod_ver)
                        
                        #one row for each band
                        csv_file.write(out_line + '\n')

            del in_raster

        csv_file.close()

        if self.bands_to_cols:
            out_csv_path_fn_reform = self.reformat(out_csv_path_fn, 'band_num', 'val')
            os.remove(out_csv_path_fn)
            os.rename(out_csv_path_fn_reform, out_csv_path_fn)
            
        del desc
        del flds
        del csv_file

    def reformat(self, fn, band_num_col, pixel_val_col):
        '''
        Reformat fn using pandas pivot table. Write reformatted data to csv.
        
        Return: reformatted CSV filename
        '''
        
        df = pd.read_csv(fn, comment='#', sep=',')  
        
        #get rid of column with no column name
        df.drop(df.filter(regex="Unname"), axis=1, inplace=True)
        
        out_fn = fn.replace('.csv', '_reform.csv')

        #include all columns except pivot  
        index_cols = [col for col in df.columns if col not in [pixel_val_col, band_num_col]]

        pt = pd.pivot_table(df, values=[pixel_val_col], index=index_cols, columns=[band_num_col])

        # #rename band number columns to mapped rhos_<wv> names
        band_map = {n:'bnum_{}'.format(n) for n in pt.columns.levels[1]}
        pt.rename(columns=band_map, inplace=True)

        pt.columns = pt.columns.droplevel(0)
        pt.columns.name = None
        pt = pt.reset_index()
        
        pt.to_csv(out_fn, index=False)    
        
        return out_fn

    def get_ext_points(self, center_pnt, cell_size, window_size):
        '''
        Returns list of points that define raster cell centers for a window of window_size.
        The window center center point is identified by center_pnt.

        Note the lower-left cell in the window is the first point in the list.
        '''
        pnts = {}

        if window_size == 1:
            pnts['0,0'] = center_pnt

        else:
            h = window_size // 2
            row = window_size-1
            for i in range(-(h), h+1):
                col = 0
                for j in range(-(h), h+1):
                    pnts['%s,%s' % (row, col)] = (arcpy.Point(center_pnt.X + (cell_size * j), center_pnt.Y + (cell_size * i)))
                    col += 1
                row -= 1

        return pnts


######################
######################
class PixelExtractByPolygon(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""

            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            return

    def __init__(self):
        super(PixelExtractByPolygon, self).__init__()
        self.label = 'Pixel extract by polygons'
        self.description = 'Extract pixels values from one or more rasters to a CSV file based on polygon feature class.'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]

        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'

        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'

        # Input_polygon_feature_class
        param4 = arcpy.Parameter()
        param4.name = 'Input_polygon_feature_class'
        param4.displayName = 'Input polygon feature class'
        param4.parameterType = 'Required'
        param4.direction = 'Input'
        param4.datatype = 'Feature Layer'

        # Polygon_field
        param5 = arcpy.Parameter()
        param5.name = 'Polygon_field'
        param5.displayName = 'Polygon attribute(s)'
        param5.parameterType = 'Optional'
        param5.direction = 'Input'
        param5.datatype = 'Field'
        param5.multiValue = True
        param5.filter.list = ['Short', 'Long', 'Single', 'Double', 'Text', 'Date']
        param5.parameterDependencies = [param4.name]

        # Minimum valid_data value
        param6 = arcpy.Parameter()
        param6.name = 'Minimum_valid_data_value'
        param6.displayName = 'Minimum valid data value'
        param6.parameterType = 'Optional'
        param6.direction = 'Input'
        param6.datatype = 'Double'
        param6.value = '1'

        # Maximum valid_data value
        param7 = arcpy.Parameter()
        param7.name = 'Maximum_valid_data_value'
        param7.displayName = 'Maximum valid data value'
        param7.parameterType = 'Optional'
        param7.direction = 'Input'
        param7.datatype = 'Double'
        param7.value = '250'

        # Out_csv_file_name
        param8 = arcpy.Parameter()
        param8.name = 'Out_csv_file_name'
        param8.displayName = 'Out CSV file name'
        param8.parameterType = 'Required'
        param8.direction = 'Output'
        param8.datatype = 'File'
        param8.filter.list = ['csv']

        return [param0, param1, param2, param3, param1b, param4, param5, param6, param7, param8]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'pixel_extract_by_polygon'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            # self.in_ws_type = self.get_ws_type(self.in_ws)
            self.in_polygon_fc = parameters[5].valueAsText
            self.poly_attribnames = parameters[6].valueAsText.split(';') if parameters[6].value else []
            self.min_valid_val = parameters[7].value
            self.max_valid_val = parameters[8].value
            full_out_csv_fn  = os.path.abspath(parameters[9].valueAsText)
            self.out_ws, self.out_csv_fn  = os.path.split(full_out_csv_fn )
            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_csv_fn  = self.validate_csv_name(self.out_csv_fn )
                
            self.get_arc_lic('Spatial')

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\nselection_method=%s\nin_ws=%s\nin_filter=%s\nrecursive=%s\nin_filelist_fn=%s\nin_polygon_fc=%s\nout_polygon_fc=%s\nout_ws=%s\n' % (self.input_method, self.in_ws, self.in_filter, self.recursive, self.in_filelist_fn, self.in_polygon_fc, self.out_csv_fn, self.out_ws))
                self.log_file.write('poly_attribnames={0}\n'.format(self.poly_attribnames))
                self.log_file.write ('min_valid_val=%s\n' % (self.min_valid_val))
                self.log_file.write ('max_valid_val=%s\n' % (self.max_valid_val))
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)

            # arcpy.env.workspace = self.in_ws

            if (arcpy.Describe(self.in_polygon_fc).shapeType != 'Polygon' and
                arcpy.Describe(self.in_polygon_fc).shapeType != 'Polyline'):
                arcpy.AddError('Input feature class must have a shape type "Polygon" or "Polyline"...exiting.')
                arcpy.AddError(f'Input feature class shape type {arcpy.Describe(self.in_polygon_fc).shapeType}...exiting.')
                return

            in_files = self.get_files()
            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            self.extract(in_files)
            # self.clean_up()
            del in_files

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)

            arcpy.env.overwriteOutput = self.orig_overwriteoutput

    def extract(self, in_files):
        '''
        Use polygon feature class to extract raster values for all raster in in_files list. Output written to
        CSV file.
        '''
        ext_raster = None
        base_fn_noext = os.path.splitext(os.path.split(self.in_polygon_fc)[1])[0]
        tmp_point_fc_fn = self.valid_fc_name('%s_extpts' % base_fn_noext, arcpy.env.scratchGDB)
        tmp_point_jn_fc_fn = self.valid_fc_name('%s_extpts_jn' % base_fn_noext, arcpy.env.scratchGDB)

        ##Setup for CSV output file
        flds = ['SHAPE@X', 'SHAPE@Y', 'grid_code']

        #assuming first file matches others
        can_unscale = self.can_unscale(in_files[0])
        if can_unscale:
            hdr = 'date,x,y,val,unscaled_val,file'
        else:
            hdr = 'date,x,y,val,file'
        
        fld_idx_poly_attrib = len(flds) 
        for poly_attribname in self.poly_attribnames:
            flds.append(poly_attribname)
            hdr += ',{0}'.format(poly_attribname)
            
        csv_file = codecs.open(os.path.join(self.out_ws, self.out_csv_fn), "w", encoding="utf-8")

        arcpy.env.overwriteOutput = True

        for fn in in_files:
            arcpy.AddMessage('Extracting pixels for %s' % fn)
            
            if self.debug: self.log_file.write ('Extracting pixels for %s\n' % fn)
            
            (year, mm, dd, hhmm, area_name, prod_name, prod_ver) = self.parse_fn(fn)
            
            if hdr is not None:
                #use the first filename to identify type of files (assuming other are the same)
                if prod_ver is not None:
                    hdr += ',prod_ver'
                    saps_prodfile = True
                else:
                    saps_prodfile = False
                    
                csv_file.write(hdr + '\n')
                hdr = None
                        
            try:
                arcpy.env.snapRaster = fn
                arcpy.env.workspace = self.out_ws
                if arcpy.Describe(fn).bandCount > 1:
                    arcpy.AddWarning('Multiband rasters are not supported. Skipping %s.' % (fn))
                    continue

                ext_raster = ExtractByMask(fn, self.in_polygon_fc)
                arcpy.RasterToPoint_conversion(ext_raster, tmp_point_fc_fn)
            except:
                arcpy.AddError("Unable to extract pixels from %s. Verify that the raster overlaps the (selected) features in the feature class %s." % (fn, self.in_polygon_fc))
                continue

            #spatial join with in_polygon_fc_full_fn
            if self.poly_attribnames:
                fieldmappings = arcpy.FieldMappings()
                fieldmappings.addTable(tmp_point_fc_fn)

                for poly_attribname in self.poly_attribnames:
                    fm = arcpy.FieldMap()
                    fm.addInputField(self.in_polygon_fc, poly_attribname)

                    fieldmappings.addFieldMap(fm)
            else:
                fieldmappings = "#"
                        
            arcpy.SpatialJoin_analysis(tmp_point_fc_fn, self.in_polygon_fc, tmp_point_jn_fc_fn, "#", "#", fieldmappings)

            #extract fields of interest to CSV file
            if self.debug: self.log_file.write ('Reformating results and writing to %s\n' % self.out_csv_fn)

            with arcpy.da.SearchCursor(tmp_point_jn_fc_fn, flds) as cursor:
                if self.debug: self.log_file.write ('cursor.fields={0}\n'.format(','.join(cursor.fields)))

                for row in cursor:
                    val = self.validate_data(row[2])
                    
                    if can_unscale:
                        unscaled_val = self.get_unscaled_val(val, prod_name, prod_ver=prod_ver)

                        out_line = '{0}-{1}-{2},{3},{4},{5},{6},{7}'.format(mm, dd, year, row[0], row[1], val, unscaled_val, fn)
                    else:
                        out_line = '{0}-{1}-{2},{3},{4},{5},{6}'.format(mm, dd, year, row[0], row[1], val, fn)
                        
                    for i, poly_attribname in enumerate(self.poly_attribnames): 
                        out_line += ',{0}'.format(row[fld_idx_poly_attrib+i])

                    if saps_prodfile:
                        #include SAPS product version
                        out_line += ',{0}'.format(prod_ver)
                        
                    csv_file.write(out_line + '\n')

            #clean up
            arcpy.Delete_management(tmp_point_fc_fn)
            arcpy.Delete_management(tmp_point_jn_fc_fn)
            del fieldmappings

        csv_file.close()

        arcpy.env.snapRaster = ""
        del ext_raster
        del csv_file


################
################
class SwapColormap(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""

            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')


        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            return

    def __init__(self):
        super(SwapColormap, self ).__init__()

        self.label = 'Swap raster colormap'
        self.description = 'Swap internal raster colormap in-place'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]

        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'

        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'

        # Colormap template raster
        param_cmap_template = arcpy.Parameter()
        param_cmap_template.name = 'Cmap_template_raster'
        param_cmap_template.displayName = 'Colormap template raster'
        param_cmap_template.parameterType = 'Optional'
        param_cmap_template.direction = 'Input'
        param_cmap_template.datatype = 'DERasterDataset'

        # Colormap clr
        param_cmap = arcpy.Parameter()
        param_cmap.name = 'cmap_file (.clr)'
        param_cmap.displayName = 'Colormap filename (.clr)'
        param_cmap.parameterType = 'Optional'
        param_cmap.direction = 'Input'
        param_cmap.datatype = 'DEFile'
        # param_cmap.filter.type = 'File'
        param_cmap.filter.list = ['clr']

        return [param0, param1, param2, param3, param1b, param_cmap_template, param_cmap]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'SwapColormap'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.cmap_template = parameters[5].valueAsText
            self.cmap_file = parameters[6].valueAsText

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\n')               
                for p in parameters:
                    self.log_file.write ('\t{0}={1}\n'.format(p.name, p.valueAsText))
                #self.log_file.write ('Parameters:\nin_ws=%s\nin_filter=%s\nrecursive=%s\ncmap_fn=%s\n' % (self.in_ws, self.in_filter, self.recursive, self.cmap_file))
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)

            in_files = self.get_files()

            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            rc = self.apply_cmap(in_files, self.cmap_template, self.cmap_file)

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
#            for extension_name in self.lic_checked_out:
#                if self.lic_checked_out[extension_name]:
#                   self.return_arc_lic(extension_name)

            arcpy.env.overwriteOutput = self.orig_overwriteoutput


    def apply_cmap(self, in_fns, cmap_template, cmap_file):
        '''
        For each file in in_fns list, apply the cmap from the cmap_template if defined else use CLR cmap_file.
        '''
            
        arcpy.env.overwriteOutput = True
        
        if cmap_template is None:
            opt = 'clr file'
            #use first raster to identify raster type
            rb_info = arcpy.Describe('%s\\Band_1' % in_fns[0])
            
            if rb_info.pixelType in ['U1', 'U2', 'U4', 'U8', 'U16']:
                
                num_bits = 2**int(rb_info.pixelType[1:])
                fixed_cmap_file = self.fix_clr(cmap_file, num_bits)
                
                if fixed_cmap_file is None:
                    return 3
            else:
                arcpy.AddError('Unsupported input raster: pixelType=%s...returning.' % rb_info.pixelType)
                if self.log_file: self.log_file.write('Adding colormap failed: Unsupported input raster: pixelType=%s...returning.\n' % rb_info.pixelType)
                return 4
        else:
            opt = 'colormap template raster'
            cmap_raster = arcpy.Raster(cmap_template)
            
            if not cmap_raster.isInteger:
                arcpy.AddError('Colormap template file does not have color map...returning.')
                if self.log_file: self.log_file.write('Adding colormap failed: Non-integer type raster template file does not support colormaps...returning.\n')
                return 1

        for fn in in_fns:
                
            arcpy.AddMessage('Applying colormap from %s to %s\n' % (opt, fn))

            try:
                #delete existing cmap if it exists
                arcpy.DeleteColormap_management(fn)
            except Exception:
                #write message to log and then ignore it
                msgs = arcpy.GetMessages(2)
                if self.log_file: self.log_file.write('Warning: Adding colormap deleteColormap failed - assuming no colormap in raster...not critical so continuing.\n')
                if self.log_file: self.log_file.write('Reported arcpy exception: %s\n' % msgs)
                pass
                
            try:            
                if cmap_template is None:
                    arcpy.AddColormap_management(fn, "#", fixed_cmap_file)
                else:
                    arcpy.AddColormap_management(fn, cmap_template, "#")                
                    
            except arcpy.ExecuteError:
                msgs = arcpy.GetMessages(2)
                arcpy.AddError('Adding colormap from %s failed...returning.' % (opt))
                arcpy.AddError(msgs)

                if self.log_file: self.log_file.write('Adding colormap failed: assuming the %s does not have one or is invalid...returning.\n' % (opt))
                if self.log_file: self.log_file.write('Reported arcpy exception: %s\n' % msgs)
                
                return 2
        
        try:
            os.remove(fixed_cmap_file)
        except Exception:
            pass

        return

    def fix_clr(self, cmap_file, num_bits):
        '''
        Ensure these an entry for each possible pixel value. Missing entries are assigned a rgb=0,0,0. 
        
        Without this the AddColormap_management (as of v10.6.1) creates the colormap
        in a <fn>.clr file rather than adding it as an internal colormap with the tif.
        
        Returns name of "fixed" colormap file
        '''
        
        try:
        
            #create default cmap (rgb=0,0,0)   
            if self.log_file: self.log_file.write('Info in fix_clr: num_bits {0}\n'.format(num_bits))            

            out_cmap = [[i, '0 0 0'] for i in range(0, num_bits)]
            
            #update cmap based on clr file
            with open(cmap_file, 'r') as fh:
            
                for ln in fh.readlines():
                    ln = ln.strip()
                    #ignore comment lines
                    if ln.startswith('#') or ln.startswith('!') or not ln:
                        continue
                        
                    x, r, g, b = ln.split()
                    
                    out_cmap[int(x)] = [x, '{0} {1} {2}'.format(r, g, b)]
                    
                    #if self.log_file: self.log_file.write('out_map[0:5]={0}\n '.format(out_cmap[0:5]))                        
            
            #create "fixed" clr file
            fixed_cmap_file = os.path.join(arcpy.env.scratchFolder, os.path.basename(cmap_file)) 
            with open(fixed_cmap_file, 'w') as fh:
                
                for i in out_cmap:
                    fh.write('{0} {1}\n'.format(i[0], i[1]))
                
            return fixed_cmap_file
            
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError('Bad format for clr file {0}'.format(cmap_file))
            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write('Adding colormap failed: Bad format for clr file {0}...returning.\n'.format(cmap_file))
            if self.log_file: self.log_file.write('Reported arcpy exception: %s\n' % msgs)

            return None
        
################
################

class ApplyCorrection(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""
            
            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            return

    def __init__(self):
        super(ApplyCorrection, self ).__init__()

        self.label = 'xApply correction to rasters (alpha)'
        self.description = 'Add/Subtract list of rasters from a "correction" raster to produce a "corrected" raster.'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]
        
        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'
        
        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'
        
        # Correction raster
        param4 = arcpy.Parameter()
        param4.name = 'Correction_raster'
        param4.displayName = 'Correction raster'
        param4.parameterType = 'Required'
        param4.direction = 'Input'
        param4.datatype = 'Raster Layer'
        
        # Correction type
        param_type = arcpy.Parameter()
        param_type.name = 'Correction_type'
        param_type.displayName = 'Correction type'
        param_type.parameterType = 'Required'
        param_type.direction = 'Input'
        param_type.datatype = 'String'
        param_type.filter.list = ['Add', 'Subtract']
        param_type.value = param_type.filter.list[1]
        
        # Minimum valid_data value
        param_min = arcpy.Parameter()
        param_min.name = 'Minimum_valid_data_value'
        param_min.displayName = 'Minimum valid data value'
        param_min.parameterType = 'Optional'
        param_min.direction = 'Input'
        param_min.datatype = 'Double'
        param_min.value = '1'

        # Maximum valid_data value
        param_max = arcpy.Parameter()
        param_max.name = 'Maximum_valid_data_value'
        param_max.displayName = 'Maximum valid data value'
        param_max.parameterType = 'Optional'
        param_max.direction = 'Input'
        param_max.datatype = 'Double'
        param_max.value = '250'

        # Nodata_value
        param_nodata = arcpy.Parameter()
        param_nodata.name = 'Nodata_value'
        param_nodata.displayName = 'Nodata value'
        param_nodata.parameterType = 'Required'
        param_nodata.direction = 'Input'
        param_nodata.datatype = 'Double'
        param_nodata.value = '255'

        # Output_filename_suffix
        param_out_suf = arcpy.Parameter()
        param_out_suf.name = 'Output_filename_suffix'
        param_out_suf.displayName = 'Output filename suffix'
        param_out_suf.parameterType = 'Optional'
        param_out_suf.direction = 'Input'
        param_out_suf.datatype = 'String'
        param_out_suf.value = '_corr'

        # Out workspace
        param_out_ws = arcpy.Parameter()
        param_out_ws.name = 'Output_workspace'
        param_out_ws.displayName = 'Output workspace'
        param_out_ws.parameterType = 'Required'
        param_out_ws.direction = 'Input'
        param_out_ws.datatype = 'Workspace'

        return [param0, param1, param2, param3, param1b, param4, param_type, param_min, param_max, param_nodata, param_out_suf, param_out_ws]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'ApplyCorrection'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.corr_raster_fn = parameters[5].valueAsText            
            self.corr_type = parameters[6].valueAsText            
            self.min_valid_val = parameters[7].value
            self.max_valid_val = parameters[8].value
            self.nodata_value = parameters[9].value          
            self.out_fn_suffix = self.validate_fn_suffix(parameters[10].valueAsText)
            self.out_ws = parameters[11].valueAsText
            
            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_ext = self.get_out_ext(self.out_ws_type)

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\ninput_method=%s\n' % self.input_method)
                self.log_file.write ('in_ws=%s\n' % self.in_ws)
                self.log_file.write ('in_filter=%s\n' % self.in_filter)
                self.log_file.write ('recursive=%s\n' % self.recursive)
                self.log_file.write ('in_filelist_fn=%s\n' % self.in_filelist_fn)
                self.log_file.write ('corr_raster_fn=%s\n' % self.corr_raster_fn)
                self.log_file.write ('corr_type=%s\n' % self.corr_type)
                self.log_file.write ('min_valid_val=%s\n' % (self.min_valid_val))
                self.log_file.write ('max_valid_val=%s\n' % (self.max_valid_val))
                self.log_file.write ('nodata_val=%s\n' % (self.nodata_value))
                self.log_file.write ('out_fn_suffix=%s\n' % self.out_fn_suffix)
                self.log_file.write ('out_ws=%s\n' % self.out_ws)
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)

            self.get_arc_lic('Spatial')

            in_files = self.get_files()

            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            corrected_fns = self.apply_correction(in_files, self.corr_type)

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)
            arcpy.env.overwriteOutput = self.orig_overwriteoutput

    def undo_masking(self, in_raster, src_raster):
        '''
        Assign masked value from source raster to target raster.
        
        Return updated raster
        '''
        if self.debug: self.log_file.write ('undo_masking: in_ras = %s\n' % (in_raster))

        #valid data values in in_raster are returned and invalid (or masked) set original value from src_raster
        unmasked_raster = Con(IsNull(in_raster), src_raster, in_raster)
        
        return unmasked_raster 
        
    def apply_correction(self, fns, corr_type):
        '''
        Subtract or Add (based on corr_type) a list of rsster from Correction raster

        Return list of output filenames
        '''
        corr_fns = []
        corr_raster = Raster(self.corr_raster_fn)

        for fn in fns:

            arcpy.AddMessage('Creating corrected raster for %s' % fn)
            if self.debug: self.log_file.write ('Creating corrected raster for %s.\n' % fn)

            fn_base, ext = os.path.splitext(os.path.split(fn)[1])
            if self.out_fn_suffix:
                fn_base += self.out_fn_suffix
            if self.out_ext:
                fn_base += ext
            else:
                fn_base = fn_base.replace('.', self.ws_delimiter)
            corr_fn = os.path.join( self.out_ws, fn_base)

            #create raster of only valid data
            masked_fn_list = self.create_masks([fn])
            masked_fn = masked_fn_list[0] if masked_fn_list else fn
            
            if corr_type == 'Subtract':
                corrected_raster = Minus(Raster(masked_fn), corr_raster)
            elif corr_type == 'Add':
                corrected_raster = Add(Raster(masked_fn), corr_raster)
            else:
                arcpy.AddError('Unsupported correction type "%s"...Exiting.' % corr_type)
                sys.exit(1)
            
            if self.debug: self.log_file.write ('done %s\n' % corr_type)
                
            src_raster = Raster(fn)
            if corr_type == 'Subtract' and self.min_valid_val:
                #set any zero values to minimum valid value
                if self.debug: self.log_file.write ('Setting 0 values to min_valid_val (=%s)\n' % self.min_valid_val)
                corrected_raster = Con((corrected_raster == 0), self.min_valid_val, corrected_raster)
            elif corr_type == 'Add' and self.max_valid_val:
                #set any values > maximum to maximum valid value
                if self.debug: self.log_file.write ('Setting values greater than max_valid_val (=%s) to max_valid_val.\n' % self.min_valid_val)
                corrected_raster = Con((corrected_raster > self.max_valid_val), self.max_valid_val, corrected_raster)
                
            #apply original masking of invalid data (i.e. SAPS cloud, mixed pixel, land, etc)
            if self.debug: self.log_file.write ('applying original non-valid data masking\n')
            corrected_raster = self.undo_masking(corrected_raster, src_raster)
            self.copy_raster(corrected_raster, corr_fn, src_raster.pixelType, self.nodata_value)
            
            if src_raster.isInteger:
                arcpy.AddColormap_management(corr_fn, fn, "#")
                
            corr_fns.append(corr_fn)

            if masked_fn != fn:
                arcpy.Delete_management(masked_fn)

            del corrected_raster
            
        del corr_raster
        
        return corr_fns

################
################

class GenerateAnomaly(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""
            
            method_pnum = 0
            self.params[method_pnum+1].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+2].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+3].enabled = (self.params[method_pnum].valueAsText == 'Workspace')
            self.params[method_pnum+4].enabled = (self.params[method_pnum].valueAsText == 'Filelist')

            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""

            method_pnum = 0
            if self.params[method_pnum].valueAsText == 'Workspace':
                for i in range(1, 4):
                    if not self.params[method_pnum+i].valueAsText:
                        self.params[method_pnum+i].setErrorMessage('ERROR: %s parameter is required if "Workspace" file selection method selected.' % (self.params[method_pnum+i].displayName))

                self.params[method_pnum+4].clearMessage()
            else:
                for i in range(1, 4):
                    self.params[method_pnum+i].clearMessage()

                if not self.params[method_pnum+4].valueAsText:
                    self.params[method_pnum+4].setErrorMessage('ERROR: %s parameter is required if "Filelist" file selection method selected.' % (self.params[method_pnum+4].displayName))

            return

    def __init__(self):
        super(GenerateAnomaly, self ).__init__()

        self.label = 'xGenerate anomaly rasters from reference raster (alpha)'
        self.description = 'Subtract list of rasters from a reference raster to produce anomaly raster.'
        self.canRunInBackground = False

    def getParameterInfo(self):

        # Input file selection method
        param0 = arcpy.Parameter()
        param0.name = 'Input_file_selection_method'
        param0.displayName = 'Input file selection method'
        param0.parameterType = 'Required'
        param0.direction = 'Input'
        param0.datatype = 'String'
        param0.filter.list = ['Workspace', 'Filelist']
        param0.value = param0.filter.list[0]
        
        # Input_workspace
        param1 = arcpy.Parameter()
        param1.name = 'Input_workspace'
        param1.displayName = 'Input workspace'
        # param1.parameterType = 'Required'
        param1.parameterType = 'Optional'
        param1.direction = 'Input'
        param1.datatype = 'Workspace'

        # Input_filename_filter
        param2 = arcpy.Parameter()
        param2.name = 'Input_filename_filter'
        param2.displayName = 'Input filename filter'
        # param2.parameterType = 'Required'
        param2.parameterType = 'Optional'
        param2.direction = 'Input'
        param2.datatype = 'String'
        param2.value = '*.tif'

        # Recursive
        param3 = arcpy.Parameter()
        param3.name = 'Recursive'
        param3.displayName = 'Recursive'
        param3.parameterType = 'Required'
        param3.direction = 'Input'
        param3.datatype = 'Boolean'
        param3.value = 'false'
        
        # Input_filelist
        param1b = arcpy.Parameter()
        param1b.name = 'Input_filelist'
        param1b.displayName = 'Input filelist'
        # param1b.parameterType = 'Required'
        param1b.parameterType = 'Optional'
        param1b.direction = 'Input'
        param1b.datatype = 'Text File'
        
        # Reference raster
        param4 = arcpy.Parameter()
        param4.name = 'Reference_raster'
        param4.displayName = 'Reference raster'
        param4.parameterType = 'Required'
        param4.direction = 'Input'
        param4.datatype = 'Raster Layer'

        # Minimum valid_data value
        param_min = arcpy.Parameter()
        param_min.name = 'Minimum_valid_data_value'
        param_min.displayName = 'Minimum valid data value'
        param_min.parameterType = 'Optional'
        param_min.direction = 'Input'
        param_min.datatype = 'Double'
        param_min.value = '1'

        # Maximum valid_data value
        param_max = arcpy.Parameter()
        param_max.name = 'Maximum_valid_data_value'
        param_max.displayName = 'Maximum valid data value'
        param_max.parameterType = 'Optional'
        param_max.direction = 'Input'
        param_max.datatype = 'Double'
        param_max.value = '250'

        # Nodata_value
        param_nodata = arcpy.Parameter()
        param_nodata.name = 'Nodata_value'
        param_nodata.displayName = 'Nodata value'
        param_nodata.parameterType = 'Required'
        param_nodata.direction = 'Input'
        param_nodata.datatype = 'Double'
        param_nodata.value = '255'

        # Output_filename_suffix
        param_out_suf = arcpy.Parameter()
        param_out_suf.name = 'Output_filename_suffix'
        param_out_suf.displayName = 'Output filename suffix'
        param_out_suf.parameterType = 'Optional'
        param_out_suf.direction = 'Input'
        param_out_suf.datatype = 'String'
        param_out_suf.value = '_anom'

        # Out workspace
        param_out_ws = arcpy.Parameter()
        param_out_ws.name = 'Output_workspace'
        param_out_ws.displayName = 'Output workspace'
        param_out_ws.parameterType = 'Required'
        param_out_ws.direction = 'Input'
        param_out_ws.datatype = 'Workspace'

        # colormap 
#        colormap_def_fn = os.path.join(os.path.dirname(sys.argv[0]), 'colormaps', 'anom_colormap.clr')
        colormap_def_fn = os.path.join(os.path.dirname(__file__), 'colormaps', 'anom_colormap.clr')
        param_cmap = arcpy.Parameter()
        param_cmap.name = 'Anomaly_CLR_colormap'
        param_cmap.displayName = 'Anomaly CLR colormap'
        param_cmap.parameterType = 'Required'
        param_cmap.direction = 'Input'
#        param_cmap.filter.type = 'File'
#        param_cmap.filter.list = ['clr']
        param_cmap.datatype = 'DEFile'
        param_cmap.value = colormap_def_fn

        return [param0, param1, param2, param3, param1b, param4, param_min, param_max, param_nodata, param_out_suf, param_out_ws, param_cmap]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:
            num_args = len(parameters)
            tool_name = 'GenerateAnomaly'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.input_method = parameters[0].valueAsText
            self.in_ws = parameters[1].valueAsText
            self.in_filter = parameters[2].valueAsText
            self.recursive = parameters[3].value
            self.in_filelist_fn = parameters[4].valueAsText
            self.ref_raster_fn = parameters[5].valueAsText            
            self.min_valid_val = parameters[6].value
            self.max_valid_val = parameters[7].value
            self.nodata_value = parameters[8].value          
            self.out_fn_suffix = self.validate_fn_suffix(parameters[9].valueAsText)
            self.out_ws = parameters[10].valueAsText
            self.anom_colormap = parameters[11].valueAsText
            self.mode = 'correction'
            
            self.out_ws_type = self.get_ws_type(self.out_ws)
            self.out_ext = self.get_out_ext(self.out_ws_type)

            if self.debug:
                self.log_file.write ('RS_tools version = %s\n' % self.getversion())
                self.log_file.write ('Parameters:\ninput_method=%s\n' % self.input_method)
                self.log_file.write ('in_ws=%s\n' % self.in_ws)
                self.log_file.write ('in_filter=%s\n' % self.in_filter)
                self.log_file.write ('recursive=%s\n' % self.recursive)
                self.log_file.write ('in_filelist_fn=%s\n' % self.in_filelist_fn)
                self.log_file.write ('ref_raster_fn=%s\n' % self.ref_raster_fn)
                self.log_file.write ('min_valid_val=%s\n' % (self.min_valid_val))
                self.log_file.write ('max_valid_val=%s\n' % (self.max_valid_val))
                self.log_file.write ('nodata_val=%s\n' % (self.nodata_value))
                self.log_file.write ('out_fn_suffix=%s\n' % self.out_fn_suffix)
                self.log_file.write ('out_ws=%s\n' % self.out_ws)
                self.log_file.write ('out ws_type=%s\n' % self.out_ws_type)
                self.log_file.write ('ScratchGDB=%s\n' % arcpy.env.scratchGDB)
                self.log_file.write ('ScratchFolder=%s\n' % arcpy.env.scratchFolder)
                self.log_file.write ('ScratchWS=%s\n' % arcpy.env.scratchWorkspace)

            self.get_arc_lic('Spatial')

            in_files = self.get_files()

            if not in_files:
                arcpy.AddWarning('No files found to process...exiting.')
                return

            anom_fns = self.make_anomalies(in_files)

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except missingRequiredParameter as e:
            arcpy.AddError('Missing required parameter "%s".' % str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)
            arcpy.env.overwriteOutput = self.orig_overwriteoutput
    
    def get_raster_max(self, in_raster, nodata_value, abs=True):
        '''
        TODO: nodata??
        '''
        arr = arcpy.RasterToNumPyArray(in_raster, nodata_to_value=0)        #nodata_value)
        max_val = np.nanmax(np.abs(arr)) if abs else np.nanmax(arr)
        del arr
        return np.asscalar(max_val)
    
    def make_anomalies(self, fns):
        '''
        Subtract list of rsster from reference raster to generate an anomaly.
        Scale values to 8-bit raster.
        
        TODO options to scaling of output

        Return list of output filenames
        '''

        anom_fns = []
        ref_raster = Raster(self.ref_raster_fn)

        for fn in fns:

            arcpy.AddMessage('Creating anomaly raster for %s' % fn)
            if self.debug: self.log_file.write ('Creating anomaly raster for %s.\n' % fn)

            fn_base, ext = os.path.splitext(os.path.split(fn)[1])
            if self.out_fn_suffix:
                fn_base += self.out_fn_suffix
            if self.out_ext:
                fn_base += ext
            else:
                fn_base = fn_base.replace('.', self.ws_delimiter)
            anom_fn = os.path.join( self.out_ws, fn_base)

            #create raster of only valid data
            masked_fn_list = self.create_masks([fn])
            masked_fn = masked_fn_list[0] if masked_fn_list else fn
            
            anom_raster = Minus(Raster(masked_fn), ref_raster)
            if self.debug: self.log_file.write ('done minus\n')
                
            # #scale raster - maintain differences between -127 to 127; any values above 
            # #     the limits are dumped into the last bin
            # anom_raster2 = Plus(anom_raster, 128)
            # if self.debug: self.log_file.write ('done plus\n')
            # anom_raster3 = Con(anom_raster2 < 0, 0, Con(anom_raster2 > 254, 254, anom_raster2))
            # if self.debug: self.log_file.write ('done Con\n')
            # self.copy_raster(anom_raster3, anom_fn, 'U8', self.nodata_value)
            
            # #alternate raster scaling - evenly distrbute differences between +- abs(max of data)
            # data_max = self.max_valid_val if self.max_valid_val else self.get_raster_max(anom_raster, self.nodata_value, abs=True)
            # if self.log_file: self.log_file.write('Max anomaaly data value for scaling=%s\n' % data_max)
            # data_max *= 1.0
            # if self.log_file: self.log_file.write('data_max type=%s\n' % type(data_max))
            # anom_raster2 = Con((anom_raster < 0), Int(((anom_raster + data_max)/data_max)*127), Con((anom_raster > 0), Int((anom_raster/data_max)*127) + 128, 128))
            # if self.debug: self.log_file.write ('done Con\n')
            # self.copy_raster(anom_raster2, anom_fn, 'U8', self.nodata_value)
            
            # if self.debug: self.log_file.write ('done copy\n')
            # arcpy.AddColormap_management(anom_fn, "#", self.anom_colormap)
            # if self.debug: self.log_file.write ('done colormap\n')
            #todo - consider making a VAT table to map actual values to colormap 

            self.copy_raster(anom_raster, anom_fn.replace(self.out_fn_suffix, '%s_diff' % self.out_fn_suffix), 'F32', self.nodata_value)

            anom_fns.append(anom_fn)

            if masked_fn != fn:
                arcpy.Delete_management(masked_fn)

            # del anom_raster2
            del anom_raster

        return anom_fns


class FilenameParserDef(RsTools):

    class ToolValidator(object):
        """Class for validating a tool's parameter values and controlling
        the behavior of the tool's dialog."""

        def __init__(self, parameters):
            """Setup arcpy and the list of tool parameters."""
            self.params = parameters

        def initializeParameters(self):
            """Refine the properties of a tool's parameters.  This method is
            called when the tool is opened."""
            return

        def updateParameters(self):
            """Modify the values and properties of parameters before internal
            validation is performed.  This method is called whenever a parameter
            has been changed."""
            #disable params if "reset" checkbox true
            for p_idx in range(1, 9):
                self.params[p_idx].enabled = not self.params[0].value
            return

        def updateMessages(self):
            """Modify the messages created by internal validation for each tool
            parameter.  This method is called after internal validation."""
            
            p_idx = 2
            if self.params[p_idx].value:
                if len(self.params[p_idx].valueAsText) > 1:
                    self.params[p_idx].setErrorMessage('ERROR: %s parameter must only be a single character.' % (self.params[p_idx].displayName))
                else:
                    self.params[p_idx].clearMessage()

            for p_idx in range(3, 9):
                if self.params[p_idx].value:
                    try:
                        l = [int(val) for val in self.params[p_idx].valueAsText.split(',')]
                        #area_name and prod_name params are optional
                        val_lens = [1, 3] if p_idx <=6 else [0, 1, 3]
                        if len(l) not in val_lens:
                            self.params[p_idx].setErrorMessage('ERROR: %s parameter must contain a comma-separated list with either 0, 1, or 3 integers.' % (self.params[p_idx].displayName))
                    except ValueError:
                        self.params[p_idx].setErrorMessage('ERROR: %s parameter must contain a comma-separated list of integers. The list may only contain 0, 1, or 3 integers.' % (self.params[p_idx].displayName))
                else:
                    self.params[p_idx].clearMessage()
            return

    def __init__(self):
        super( FilenameParserDef, self ).__init__()
        self.label = 'xDefine filename parsing format (alpha)'
        self.description = 'Define custom filenaming parsing format for L3 files.'
        self.canRunInBackground = False

        self.initialize_defaults()

    def initialize_defaults(self):
        '''
        '''
        # fn_parser_def = self.load_fn_parser_def()
        fn_parser_def = self.fn_parser_def
        
        self.name = fn_parser_def['name']
        self.delim = fn_parser_def['delim']
        self.year = ','.join([str(i) for i in fn_parser_def['year']])
        self.mm = ','.join([str(i) for i in fn_parser_def['mm']])
        self.dd = ','.join([str(i) for i in fn_parser_def['dd']])
        self.hhmm = ','.join([str(i) for i in fn_parser_def['hhmm']])
        self.area_name = ','.join([str(i) for i in fn_parser_def['area_name']])
        self.prod_name = ','.join([str(i) for i in fn_parser_def['prod_name']])
        self.ver_info = ','.join([str(i) for i in fn_parser_def['ver_info']])
        
    def getParameterInfo(self):

        # reset
        param_reset = arcpy.Parameter()
        param_reset.name = 'Reset_defaults'
        param_reset.displayName = 'Reset defaults'
        param_reset.parameterType = 'Optional'
        param_reset.direction = 'Input'
        param_reset.datatype = 'Boolean'
        param_reset.value = 'false'

        # name
        param_name = arcpy.Parameter()
        param_name.name = 'Custom_format_name'
        param_name.displayName = 'Custom format name'
        param_name.parameterType = 'Required'
        param_name.direction = 'Input'
        param_name.datatype = 'String'

        # delimiter
        param_delim = arcpy.Parameter()
        param_delim.name = 'Filename_delimiter'
        param_delim.displayName = 'Filename delimiter'
        param_delim.parameterType = 'Required'
        param_delim.direction = 'Input'
        param_delim.datatype = 'String'

        # year
        param_year = arcpy.Parameter()
        param_year.name = 'Year_definition'
        param_year.displayName = 'Year definition'
        param_year.parameterType = 'Required'
        param_year.direction = 'Input'
        param_year.datatype = 'String'

        # mm
        param_mm = arcpy.Parameter()
        param_mm.name = 'month_definition'
        param_mm.displayName = 'Month definition (mm)'
        param_mm.parameterType = 'Required'
        param_mm.direction = 'Input'
        param_mm.datatype = 'String'
        
        # dd
        param_dd = arcpy.Parameter()
        param_dd.name = 'Day_definition'
        param_dd.displayName = 'Day definition (dd)'
        param_dd.parameterType = 'Required'
        param_dd.direction = 'Input'
        param_dd.datatype = 'String'

        # hhmm
        param_hhmm = arcpy.Parameter()
        param_hhmm.name = 'starttime_definition'
        param_hhmm.displayName = 'Starttime definition (hhmm)'
        param_hhmm.parameterType = 'Required'
        param_hhmm.direction = 'Input'
        param_hhmm.datatype = 'String'

        # area_name
        param_area_name = arcpy.Parameter()
        param_area_name.name = 'area_name_definition'
        param_area_name.displayName = 'Area name definition'
        param_area_name.parameterType = 'Optional'
        param_area_name.direction = 'Input'
        param_area_name.datatype = 'String'

        # prod_name
        param_prod_name = arcpy.Parameter()
        param_prod_name.name = 'prod_name_definition'
        param_prod_name.displayName = 'Product name definition'
        param_prod_name.parameterType = 'Optional'
        param_prod_name.direction = 'Input'
        param_prod_name.datatype = 'String'

        return [param_reset, param_name, param_delim, param_year, param_mm, param_dd, param_hhmm, param_area_name, param_prod_name]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        p_idx = 1
        if not parameters[p_idx].value:
            self.initialize_defaults()
            parameters[p_idx].value = self.name
            parameters[p_idx + 1].value = self.delim
            parameters[p_idx + 2].value = self.year
            parameters[p_idx + 3].value = self.mm
            parameters[p_idx + 4].value = self.dd
            parameters[p_idx + 5].value = self.hhmm
            parameters[p_idx + 6].value = self.area_name
            parameters[p_idx + 7].value = self.prod_name
        
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateParameters()

    def updateMessages(self, parameters):
        validator = getattr(self, 'ToolValidator', None)
        if validator:
             return validator(parameters).updateMessages()

    def execute(self, parameters, messages):
        try:

            num_args = len(parameters)
            tool_name = 'filename_definition'
            arcpy.AddMessage('tool name=%s' % tool_name)

            self.setup(tool_name)

            #Read parameters
            self.reset = parameters[0].value
            self.name = parameters[1].valueAsText
            self.delim = parameters[2].valueAsText
            self.year = parameters[3].valueAsText
            self.mm = parameters[4].valueAsText
            self.dd = parameters[5].valueAsText
            self.hhmm = parameters[6].valueAsText
            self.area_name = parameters[7].value
            self.prod_name = parameters[8].value 

            if self.debug:
                self.log_file.write ('Parameters:\nreset=%s\nname=%s\ndelim=%s\nyear=%s\nmm=%s\ndd=%s\nhhmm=%s\narea_name=%s\nprod_name=%s\n' % (self.reset, self.name, self.delim, self.year, self.mm, self.dd, self.hhmm, self.area_name, self.prod_name))

            arcpy.env.overwriteOutput = True
            
            if self.reset:
                if os.path.exists(self.config_fn):
                    os.remove(self.config_fn)
                    arcpy.AddMessage('Reset filename parsing definition to default.\n')
                    if self.log_file: self.log_file.write('Reset filename parsing definition. Removed config_fn=%s\n' % self.config_fn)
            else:
                try:
                    s = shelve.open(self.config_fn, writeback=True)
                    if 'fn_parser_def' in s: self.log_file.write('1 s=%s\n' % s['fn_parser_def'])
                    s['fn_parser_def'] = { 'name': self.name, 
                                           'delim': self.delim, 
                                           'year': [int(i) for i in self.year.split(',')],
                                           'mm': [int(i) for i in self.mm.split(',')],
                                           'dd': [int(i) for i in self.dd.split(',')],
                                           'hhmm': [int(i) for i in self.hhmm.split(',')],
                                           'area_name': [int(i) for i in self.area_name.split(',')] if self.area_name else '',
                                           'prod_name': [int(i) for i in self.prod_name.split(',')] if self.prod_name else '' }
                                    
                    if 'fn_parser_def' in s: self.log_file.write('2 s=%s\n' % s['fn_parser_def'])
                except arcpy.ExecuteError:
                    msgs = arcpy.GetMessages(2)

                    arcpy.AddError(msgs)
                    if self.log_file: self.log_file.write(msgs)                    
                finally:
                    s.close()
                
                    arcpy.AddMessage('Updated filename parsing definition in config_fn=%s\n' % self.config_fn)
                    if self.log_file: self.log_file.write('Updated filename parsing definition in config_fn=%s\n' % self.config_fn)

        except overwriteError as e:
            arcpy.AddIDMessage("Error", 12, str(e))
        except arcpy.ExecuteError:
            msgs = arcpy.GetMessages(2)

            arcpy.AddError(msgs)
            if self.log_file: self.log_file.write(msgs)
        except:
            self.report_error()
        finally:
            if self.log_file: self.log_file.close()
            for extension_name in self.lic_checked_out:
                if self.lic_checked_out[extension_name]:
                    self.return_arc_lic(extension_name)
            arcpy.env.overwriteOutput = self.orig_overwriteoutput      